" Maintainer:   Yukihiro Nakadaira <yukihiro.nakadaira@gmail.com>
" License:      This file is placed in the public domain.
" Last Change:  2016-11-24

if &encoding != "utf-8" && !has("iconv")
  " Use echomsg because echoerr is not displayed in formatexpr context.
  " Echomsg is same but message is logged in :messages buffer.
  echomsg "autofmt/unicode.vim requires iconv() in non utf-8 encoding"
  echoerr "autofmt/unicode.vim requires iconv() in non utf-8 encoding"
  finish
endif

let s:cpo_save = &cpo
set cpo&vim

function autofmt#unicode#import()
  return s:lib
endfunction

if v:version > 702 || v:version == 702 && has('patch780')
  let s:char2nr_utf8 = function('char2nr')
else
  function s:char2nr_utf8(expr, utf8)
    if a:utf8 && &encoding != 'utf-8'
      let x = map(range(len(a:expr)), 'char2nr(a:expr[v:val])')
      if len(x) == 0
        return -1
      elseif len(x) == 1
        return x[0]
      elseif len(x) == 2
        return ((x[0] % 0x20) * 0x40)
              \ + (x[1] % 0x40)
      elseif len(x) == 3
        return ((x[0] % 0x10) * 0x1000)
              \ + ((x[1] % 0x40) * 0x40)
              \ + (x[2] % 0x40)
      elseif len(x) >= 4
        return ((x[0] % 0x8) * 0x40000)
              \ + ((x[1] % 0x40) * 0x1000)
              \ + ((x[2] % 0x40) * 0x40)
              \ + (x[3] % 0x40)
      endif
    else
      return char2nr(a:expr)
    endif
  endfunction
endif

let s:lib = {}

let s:lib.lb_cache = []

function s:lib.prop_line_break(char)
  let ucs4 = self.to_ucs4(a:char)
  if ucs4 < 0x10000
    let row = self.linebreak_bmp[ucs4 / 0x100]
    return len(row) == 1 ? row[0] : row[ucs4 % 0x100]
  endif
  return self.prop_search(ucs4, "XX", self.linebreak_table, self.lb_cache)
endfunction

function s:lib.prop_search(ucs4, default, table, cache)
  let item = self.prop_bsearch(a:ucs4, a:cache)
  if item != []
    return item[2]
  endif
  let item = self.prop_bsearch(a:ucs4, a:table)
  if item == []
    return a:default
  endif
  for i in range(len(a:cache) + 1)
    if i == len(a:cache) || item[1] < a:cache[i][0]
      call insert(a:cache, item, i)
      break
    endif
  endfor
  return item[2]
endfunction

function s:lib.prop_bsearch(ucs4, table)
  let [left, right] = [0, len(a:table)]
  while left < right
    let mid = (left + right) / 2
    let item = a:table[mid]
    if a:ucs4 < item[0]
      let right = mid
    elseif item[1] < a:ucs4
      let left = mid + 1
    else
      return item
    endif
  endwhile
  return []
endfunction

function s:lib.uax14_pair_table(before, after)
  " these are not handled in the pair tables
  let not_in_table = '\vAI|BK|CB|CJ|CR|LF|NL|SA|SG|SP|XX'
  if a:before =~ not_in_table || a:after =~ not_in_table
    return self.PROHIBITED_BRK
  else
    let b = self.uax14_pair_table_cols[a:before]
    let a = self.uax14_pair_table_cols[a:after]
    return self.uax14_pair_table_rows[b][a]
  endif
endfunction

function s:lib.to_ucs4(c)
  if &encoding == 'utf-8'
    return char2nr(a:c)
  else
    let u = iconv(a:c, &encoding, "utf-8")
    return s:char2nr_utf8(u, 1)
  endif
endfunction

" Line Breaking Algorithm (B=before, A=after)
" B ! A   Mandatory break at the indicated position
" B × A  No break allowed at the indicated position
" B ÷ A  Break allowed at the indicated position
" B % A   same as (B SP+ ÷ A)
"
" Table notation
" ^   (B SP* × A)
" %   (B × A) and (B SP+ ÷ A)
" @   (B SP* × A), where A is of class CM
" #   (B × A) and (B SP+ ÷ A), where A is of class CM
" _   (B ÷ A)

let s:lib.DIRECT_BRK               = '_'
let s:lib.INDIRECT_BRK             = '%'
let s:lib.COMBINING_INDIRECT_BRK   = '#'
let s:lib.COMBINING_PROHIBITED_BRK = '@'
let s:lib.PROHIBITED_BRK           = '^'
let s:lib.EXPLICIT_BRK             = '!'

" Unicode Standard Annex #14
" Unicode Line Breaking Algorithm
" Revision 37
" Table 2. Example Pair Table
let s:cols = split("OP CL CP QU GL NS EX SY IS PR PO NU AL HL ID IN HY BA BB B2 ZW CM WJ H2 H3 JL JV JT RI EB EM ZWJ")
let s:lib.uax14_pair_table_cols = {}
for s:i in range(len(s:cols))
  let s:lib.uax14_pair_table_cols[s:cols[s:i]] = s:i
endfor
unlet s:cols s:i

" table[before][after]
let s:lib.uax14_pair_table_rows = [
      \ "^^^^^^^^^^^^^^^^^^^^^@^^^^^^^^^^",
      \ "_^^%%^^^^%%_____%%__^#^________%",
      \ "_^^%%^^^^%%%%%__%%__^#^________%",
      \ "^^^%%%^^^%%%%%%%%%%%^#^%%%%%%%%%",
      \ "%^^%%%^^^%%%%%%%%%%%^#^%%%%%%%%%",
      \ "_^^%%%^^^_______%%__^#^________%",
      \ "_^^%%%^^^______%%%__^#^________%",
      \ "_^^%%%^^^__%_%__%%__^#^________%",
      \ "_^^%%%^^^__%%%__%%__^#^________%",
      \ "%^^%%%^^^__%%%%_%%__^#^%%%%%_%%%",
      \ "%^^%%%^^^__%%%__%%__^#^________%",
      \ "%^^%%%^^^%%%%%_%%%__^#^________%",
      \ "%^^%%%^^^%%%%%_%%%__^#^________%",
      \ "%^^%%%^^^%%%%%_%%%__^#^________%",
      \ "_^^%%%^^^_%____%%%__^#^________%",
      \ "_^^%%%^^^______%%%__^#^________%",
      \ "_^^%_%^^^__%____%%__^#^________%",
      \ "_^^%_%^^^_______%%__^#^________%",
      \ "%^^%%%^^^%%%%%%%%%%%^#^%%%%%%%%%",
      \ "_^^%%%^^^_______%%_^^#^________%",
      \ "____________________^___________",
      \ "%^^%%%^^^%%%%%_%%%__^#^________%",
      \ "%^^%%%^^^%%%%%%%%%%%^#^%%%%%%%%%",
      \ "_^^%%%^^^_%____%%%__^#^___%%___%",
      \ "_^^%%%^^^_%____%%%__^#^____%___%",
      \ "_^^%%%^^^_%____%%%__^#^%%%%____%",
      \ "_^^%%%^^^_%____%%%__^#^___%%___%",
      \ "_^^%%%^^^_%____%%%__^#^____%___%",
      \ "_^^%%%^^^_______%%__^#^_____%__%",
      \ "_^^%%%^^^_%____%%%__^#^_______%%",
      \ "_^^%%%^^^_%____%%%__^#^________%",
      \ "_^^%%%^^^_____%_%%__^#^______%%%",
      \ ]


" LineBreak-9.0.0.txt
" Date: 2016-05-26, 01:00:00 GMT [KW, LI]
let s:tmp = []
call extend(s:tmp, [[0x0000,0x0008,'CM'],[0x0009,0x0009,'BA'],[0x000A,0x000A,'LF']])
call extend(s:tmp, [[0x000B,0x000C,'BK'],[0x000D,0x000D,'CR'],[0x000E,0x001F,'CM']])
call extend(s:tmp, [[0x0020,0x0020,'SP'],[0x0021,0x0021,'EX'],[0x0022,0x0022,'QU']])
call extend(s:tmp, [[0x0023,0x0023,'AL'],[0x0024,0x0024,'PR'],[0x0025,0x0025,'PO']])
call extend(s:tmp, [[0x0026,0x0026,'AL'],[0x0027,0x0027,'QU'],[0x0028,0x0028,'OP']])
call extend(s:tmp, [[0x0029,0x0029,'CP'],[0x002A,0x002A,'AL'],[0x002B,0x002B,'PR']])
call extend(s:tmp, [[0x002C,0x002C,'IS'],[0x002D,0x002D,'HY'],[0x002E,0x002E,'IS']])
call extend(s:tmp, [[0x002F,0x002F,'SY'],[0x0030,0x0039,'NU'],[0x003A,0x003B,'IS']])
call extend(s:tmp, [[0x003C,0x003E,'AL'],[0x003F,0x003F,'EX'],[0x0040,0x005A,'AL']])
call extend(s:tmp, [[0x005B,0x005B,'OP'],[0x005C,0x005C,'PR'],[0x005D,0x005D,'CP']])
call extend(s:tmp, [[0x005E,0x007A,'AL'],[0x007B,0x007B,'OP'],[0x007C,0x007C,'BA']])
call extend(s:tmp, [[0x007D,0x007D,'CL'],[0x007E,0x007E,'AL'],[0x007F,0x0084,'CM']])
call extend(s:tmp, [[0x0085,0x0085,'NL'],[0x0086,0x009F,'CM'],[0x00A0,0x00A0,'GL']])
call extend(s:tmp, [[0x00A1,0x00A1,'OP'],[0x00A2,0x00A2,'PO'],[0x00A3,0x00A5,'PR']])
call extend(s:tmp, [[0x00A6,0x00A6,'AL'],[0x00A7,0x00A8,'AI'],[0x00A9,0x00A9,'AL']])
call extend(s:tmp, [[0x00AA,0x00AA,'AI'],[0x00AB,0x00AB,'QU'],[0x00AC,0x00AC,'AL']])
call extend(s:tmp, [[0x00AD,0x00AD,'BA'],[0x00AE,0x00AF,'AL'],[0x00B0,0x00B0,'PO']])
call extend(s:tmp, [[0x00B1,0x00B1,'PR'],[0x00B2,0x00B3,'AI'],[0x00B4,0x00B4,'BB']])
call extend(s:tmp, [[0x00B5,0x00B5,'AL'],[0x00B6,0x00BA,'AI'],[0x00BB,0x00BB,'QU']])
call extend(s:tmp, [[0x00BC,0x00BE,'AI'],[0x00BF,0x00BF,'OP'],[0x00C0,0x00D6,'AL']])
call extend(s:tmp, [[0x00D7,0x00D7,'AI'],[0x00D8,0x00F6,'AL'],[0x00F7,0x00F7,'AI']])
call extend(s:tmp, [[0x00F8,0x02C6,'AL'],[0x02C7,0x02C7,'AI'],[0x02C8,0x02C8,'BB']])
call extend(s:tmp, [[0x02C9,0x02CB,'AI'],[0x02CC,0x02CC,'BB'],[0x02CD,0x02CD,'AI']])
call extend(s:tmp, [[0x02CE,0x02CF,'AL'],[0x02D0,0x02D0,'AI'],[0x02D1,0x02D7,'AL']])
call extend(s:tmp, [[0x02D8,0x02DB,'AI'],[0x02DC,0x02DC,'AL'],[0x02DD,0x02DD,'AI']])
call extend(s:tmp, [[0x02DE,0x02DE,'AL'],[0x02DF,0x02DF,'BB'],[0x02E0,0x02FF,'AL']])
call extend(s:tmp, [[0x0300,0x034E,'CM'],[0x034F,0x034F,'GL'],[0x0350,0x035B,'CM']])
call extend(s:tmp, [[0x035C,0x0362,'GL'],[0x0363,0x036F,'CM'],[0x0370,0x0377,'AL']])
call extend(s:tmp, [[0x037A,0x037D,'AL'],[0x037E,0x037E,'IS'],[0x037F,0x037F,'AL']])
call extend(s:tmp, [[0x0384,0x038A,'AL'],[0x038C,0x038C,'AL'],[0x038E,0x03A1,'AL']])
call extend(s:tmp, [[0x03A3,0x0482,'AL'],[0x0483,0x0489,'CM'],[0x048A,0x052F,'AL']])
call extend(s:tmp, [[0x0531,0x0556,'AL'],[0x0559,0x055F,'AL'],[0x0561,0x0587,'AL']])
call extend(s:tmp, [[0x0589,0x0589,'IS'],[0x058A,0x058A,'BA'],[0x058D,0x058E,'AL']])
call extend(s:tmp, [[0x058F,0x058F,'PR'],[0x0591,0x05BD,'CM'],[0x05BE,0x05BE,'BA']])
call extend(s:tmp, [[0x05BF,0x05BF,'CM'],[0x05C0,0x05C0,'AL'],[0x05C1,0x05C2,'CM']])
call extend(s:tmp, [[0x05C3,0x05C3,'AL'],[0x05C4,0x05C5,'CM'],[0x05C6,0x05C6,'EX']])
call extend(s:tmp, [[0x05C7,0x05C7,'CM'],[0x05D0,0x05EA,'HL'],[0x05F0,0x05F2,'HL']])
call extend(s:tmp, [[0x05F3,0x05F4,'AL'],[0x0600,0x0608,'AL'],[0x0609,0x060B,'PO']])
call extend(s:tmp, [[0x060C,0x060D,'IS'],[0x060E,0x060F,'AL'],[0x0610,0x061A,'CM']])
call extend(s:tmp, [[0x061B,0x061B,'EX'],[0x061C,0x061C,'CM'],[0x061E,0x061F,'EX']])
call extend(s:tmp, [[0x0620,0x064A,'AL'],[0x064B,0x065F,'CM'],[0x0660,0x0669,'NU']])
call extend(s:tmp, [[0x066A,0x066A,'PO'],[0x066B,0x066C,'NU'],[0x066D,0x066F,'AL']])
call extend(s:tmp, [[0x0670,0x0670,'CM'],[0x0671,0x06D3,'AL'],[0x06D4,0x06D4,'EX']])
call extend(s:tmp, [[0x06D5,0x06D5,'AL'],[0x06D6,0x06DC,'CM'],[0x06DD,0x06DE,'AL']])
call extend(s:tmp, [[0x06DF,0x06E4,'CM'],[0x06E5,0x06E6,'AL'],[0x06E7,0x06E8,'CM']])
call extend(s:tmp, [[0x06E9,0x06E9,'AL'],[0x06EA,0x06ED,'CM'],[0x06EE,0x06EF,'AL']])
call extend(s:tmp, [[0x06F0,0x06F9,'NU'],[0x06FA,0x070D,'AL'],[0x070F,0x0710,'AL']])
call extend(s:tmp, [[0x0711,0x0711,'CM'],[0x0712,0x072F,'AL'],[0x0730,0x074A,'CM']])
call extend(s:tmp, [[0x074D,0x07A5,'AL'],[0x07A6,0x07B0,'CM'],[0x07B1,0x07B1,'AL']])
call extend(s:tmp, [[0x07C0,0x07C9,'NU'],[0x07CA,0x07EA,'AL'],[0x07EB,0x07F3,'CM']])
call extend(s:tmp, [[0x07F4,0x07F7,'AL'],[0x07F8,0x07F8,'IS'],[0x07F9,0x07F9,'EX']])
call extend(s:tmp, [[0x07FA,0x07FA,'AL'],[0x0800,0x0815,'AL'],[0x0816,0x0819,'CM']])
call extend(s:tmp, [[0x081A,0x081A,'AL'],[0x081B,0x0823,'CM'],[0x0824,0x0824,'AL']])
call extend(s:tmp, [[0x0825,0x0827,'CM'],[0x0828,0x0828,'AL'],[0x0829,0x082D,'CM']])
call extend(s:tmp, [[0x0830,0x083E,'AL'],[0x0840,0x0858,'AL'],[0x0859,0x085B,'CM']])
call extend(s:tmp, [[0x085E,0x085E,'AL'],[0x08A0,0x08B4,'AL'],[0x08B6,0x08BD,'AL']])
call extend(s:tmp, [[0x08D4,0x08E1,'CM'],[0x08E2,0x08E2,'AL'],[0x08E3,0x0903,'CM']])
call extend(s:tmp, [[0x0904,0x0939,'AL'],[0x093A,0x093C,'CM'],[0x093D,0x093D,'AL']])
call extend(s:tmp, [[0x093E,0x094F,'CM'],[0x0950,0x0950,'AL'],[0x0951,0x0957,'CM']])
call extend(s:tmp, [[0x0958,0x0961,'AL'],[0x0962,0x0963,'CM'],[0x0964,0x0965,'BA']])
call extend(s:tmp, [[0x0966,0x096F,'NU'],[0x0970,0x0980,'AL'],[0x0981,0x0983,'CM']])
call extend(s:tmp, [[0x0985,0x098C,'AL'],[0x098F,0x0990,'AL'],[0x0993,0x09A8,'AL']])
call extend(s:tmp, [[0x09AA,0x09B0,'AL'],[0x09B2,0x09B2,'AL'],[0x09B6,0x09B9,'AL']])
call extend(s:tmp, [[0x09BC,0x09BC,'CM'],[0x09BD,0x09BD,'AL'],[0x09BE,0x09C4,'CM']])
call extend(s:tmp, [[0x09C7,0x09C8,'CM'],[0x09CB,0x09CD,'CM'],[0x09CE,0x09CE,'AL']])
call extend(s:tmp, [[0x09D7,0x09D7,'CM'],[0x09DC,0x09DD,'AL'],[0x09DF,0x09E1,'AL']])
call extend(s:tmp, [[0x09E2,0x09E3,'CM'],[0x09E6,0x09EF,'NU'],[0x09F0,0x09F1,'AL']])
call extend(s:tmp, [[0x09F2,0x09F3,'PO'],[0x09F4,0x09F8,'AL'],[0x09F9,0x09F9,'PO']])
call extend(s:tmp, [[0x09FA,0x09FA,'AL'],[0x09FB,0x09FB,'PR'],[0x0A01,0x0A03,'CM']])
call extend(s:tmp, [[0x0A05,0x0A0A,'AL'],[0x0A0F,0x0A10,'AL'],[0x0A13,0x0A28,'AL']])
call extend(s:tmp, [[0x0A2A,0x0A30,'AL'],[0x0A32,0x0A33,'AL'],[0x0A35,0x0A36,'AL']])
call extend(s:tmp, [[0x0A38,0x0A39,'AL'],[0x0A3C,0x0A3C,'CM'],[0x0A3E,0x0A42,'CM']])
call extend(s:tmp, [[0x0A47,0x0A48,'CM'],[0x0A4B,0x0A4D,'CM'],[0x0A51,0x0A51,'CM']])
call extend(s:tmp, [[0x0A59,0x0A5C,'AL'],[0x0A5E,0x0A5E,'AL'],[0x0A66,0x0A6F,'NU']])
call extend(s:tmp, [[0x0A70,0x0A71,'CM'],[0x0A72,0x0A74,'AL'],[0x0A75,0x0A75,'CM']])
call extend(s:tmp, [[0x0A81,0x0A83,'CM'],[0x0A85,0x0A8D,'AL'],[0x0A8F,0x0A91,'AL']])
call extend(s:tmp, [[0x0A93,0x0AA8,'AL'],[0x0AAA,0x0AB0,'AL'],[0x0AB2,0x0AB3,'AL']])
call extend(s:tmp, [[0x0AB5,0x0AB9,'AL'],[0x0ABC,0x0ABC,'CM'],[0x0ABD,0x0ABD,'AL']])
call extend(s:tmp, [[0x0ABE,0x0AC5,'CM'],[0x0AC7,0x0AC9,'CM'],[0x0ACB,0x0ACD,'CM']])
call extend(s:tmp, [[0x0AD0,0x0AD0,'AL'],[0x0AE0,0x0AE1,'AL'],[0x0AE2,0x0AE3,'CM']])
call extend(s:tmp, [[0x0AE6,0x0AEF,'NU'],[0x0AF0,0x0AF0,'AL'],[0x0AF1,0x0AF1,'PR']])
call extend(s:tmp, [[0x0AF9,0x0AF9,'AL'],[0x0B01,0x0B03,'CM'],[0x0B05,0x0B0C,'AL']])
call extend(s:tmp, [[0x0B0F,0x0B10,'AL'],[0x0B13,0x0B28,'AL'],[0x0B2A,0x0B30,'AL']])
call extend(s:tmp, [[0x0B32,0x0B33,'AL'],[0x0B35,0x0B39,'AL'],[0x0B3C,0x0B3C,'CM']])
call extend(s:tmp, [[0x0B3D,0x0B3D,'AL'],[0x0B3E,0x0B44,'CM'],[0x0B47,0x0B48,'CM']])
call extend(s:tmp, [[0x0B4B,0x0B4D,'CM'],[0x0B56,0x0B57,'CM'],[0x0B5C,0x0B5D,'AL']])
call extend(s:tmp, [[0x0B5F,0x0B61,'AL'],[0x0B62,0x0B63,'CM'],[0x0B66,0x0B6F,'NU']])
call extend(s:tmp, [[0x0B70,0x0B77,'AL'],[0x0B82,0x0B82,'CM'],[0x0B83,0x0B83,'AL']])
call extend(s:tmp, [[0x0B85,0x0B8A,'AL'],[0x0B8E,0x0B90,'AL'],[0x0B92,0x0B95,'AL']])
call extend(s:tmp, [[0x0B99,0x0B9A,'AL'],[0x0B9C,0x0B9C,'AL'],[0x0B9E,0x0B9F,'AL']])
call extend(s:tmp, [[0x0BA3,0x0BA4,'AL'],[0x0BA8,0x0BAA,'AL'],[0x0BAE,0x0BB9,'AL']])
call extend(s:tmp, [[0x0BBE,0x0BC2,'CM'],[0x0BC6,0x0BC8,'CM'],[0x0BCA,0x0BCD,'CM']])
call extend(s:tmp, [[0x0BD0,0x0BD0,'AL'],[0x0BD7,0x0BD7,'CM'],[0x0BE6,0x0BEF,'NU']])
call extend(s:tmp, [[0x0BF0,0x0BF8,'AL'],[0x0BF9,0x0BF9,'PR'],[0x0BFA,0x0BFA,'AL']])
call extend(s:tmp, [[0x0C00,0x0C03,'CM'],[0x0C05,0x0C0C,'AL'],[0x0C0E,0x0C10,'AL']])
call extend(s:tmp, [[0x0C12,0x0C28,'AL'],[0x0C2A,0x0C39,'AL'],[0x0C3D,0x0C3D,'AL']])
call extend(s:tmp, [[0x0C3E,0x0C44,'CM'],[0x0C46,0x0C48,'CM'],[0x0C4A,0x0C4D,'CM']])
call extend(s:tmp, [[0x0C55,0x0C56,'CM'],[0x0C58,0x0C5A,'AL'],[0x0C60,0x0C61,'AL']])
call extend(s:tmp, [[0x0C62,0x0C63,'CM'],[0x0C66,0x0C6F,'NU'],[0x0C78,0x0C80,'AL']])
call extend(s:tmp, [[0x0C81,0x0C83,'CM'],[0x0C85,0x0C8C,'AL'],[0x0C8E,0x0C90,'AL']])
call extend(s:tmp, [[0x0C92,0x0CA8,'AL'],[0x0CAA,0x0CB3,'AL'],[0x0CB5,0x0CB9,'AL']])
call extend(s:tmp, [[0x0CBC,0x0CBC,'CM'],[0x0CBD,0x0CBD,'AL'],[0x0CBE,0x0CC4,'CM']])
call extend(s:tmp, [[0x0CC6,0x0CC8,'CM'],[0x0CCA,0x0CCD,'CM'],[0x0CD5,0x0CD6,'CM']])
call extend(s:tmp, [[0x0CDE,0x0CDE,'AL'],[0x0CE0,0x0CE1,'AL'],[0x0CE2,0x0CE3,'CM']])
call extend(s:tmp, [[0x0CE6,0x0CEF,'NU'],[0x0CF1,0x0CF2,'AL'],[0x0D01,0x0D03,'CM']])
call extend(s:tmp, [[0x0D05,0x0D0C,'AL'],[0x0D0E,0x0D10,'AL'],[0x0D12,0x0D3A,'AL']])
call extend(s:tmp, [[0x0D3D,0x0D3D,'AL'],[0x0D3E,0x0D44,'CM'],[0x0D46,0x0D48,'CM']])
call extend(s:tmp, [[0x0D4A,0x0D4D,'CM'],[0x0D4E,0x0D4F,'AL'],[0x0D54,0x0D56,'AL']])
call extend(s:tmp, [[0x0D57,0x0D57,'CM'],[0x0D58,0x0D61,'AL'],[0x0D62,0x0D63,'CM']])
call extend(s:tmp, [[0x0D66,0x0D6F,'NU'],[0x0D70,0x0D78,'AL'],[0x0D79,0x0D79,'PO']])
call extend(s:tmp, [[0x0D7A,0x0D7F,'AL'],[0x0D82,0x0D83,'CM'],[0x0D85,0x0D96,'AL']])
call extend(s:tmp, [[0x0D9A,0x0DB1,'AL'],[0x0DB3,0x0DBB,'AL'],[0x0DBD,0x0DBD,'AL']])
call extend(s:tmp, [[0x0DC0,0x0DC6,'AL'],[0x0DCA,0x0DCA,'CM'],[0x0DCF,0x0DD4,'CM']])
call extend(s:tmp, [[0x0DD6,0x0DD6,'CM'],[0x0DD8,0x0DDF,'CM'],[0x0DE6,0x0DEF,'NU']])
call extend(s:tmp, [[0x0DF2,0x0DF3,'CM'],[0x0DF4,0x0DF4,'AL'],[0x0E01,0x0E3A,'SA']])
call extend(s:tmp, [[0x0E3F,0x0E3F,'PR'],[0x0E40,0x0E4E,'SA'],[0x0E4F,0x0E4F,'AL']])
call extend(s:tmp, [[0x0E50,0x0E59,'NU'],[0x0E5A,0x0E5B,'BA'],[0x0E81,0x0E82,'SA']])
call extend(s:tmp, [[0x0E84,0x0E84,'SA'],[0x0E87,0x0E88,'SA'],[0x0E8A,0x0E8A,'SA']])
call extend(s:tmp, [[0x0E8D,0x0E8D,'SA'],[0x0E94,0x0E97,'SA'],[0x0E99,0x0E9F,'SA']])
call extend(s:tmp, [[0x0EA1,0x0EA3,'SA'],[0x0EA5,0x0EA5,'SA'],[0x0EA7,0x0EA7,'SA']])
call extend(s:tmp, [[0x0EAA,0x0EAB,'SA'],[0x0EAD,0x0EB9,'SA'],[0x0EBB,0x0EBD,'SA']])
call extend(s:tmp, [[0x0EC0,0x0EC4,'SA'],[0x0EC6,0x0EC6,'SA'],[0x0EC8,0x0ECD,'SA']])
call extend(s:tmp, [[0x0ED0,0x0ED9,'NU'],[0x0EDC,0x0EDF,'SA'],[0x0F00,0x0F00,'AL']])
call extend(s:tmp, [[0x0F01,0x0F04,'BB'],[0x0F05,0x0F05,'AL'],[0x0F06,0x0F07,'BB']])
call extend(s:tmp, [[0x0F08,0x0F08,'GL'],[0x0F09,0x0F0A,'BB'],[0x0F0B,0x0F0B,'BA']])
call extend(s:tmp, [[0x0F0C,0x0F0C,'GL'],[0x0F0D,0x0F11,'EX'],[0x0F12,0x0F12,'GL']])
call extend(s:tmp, [[0x0F13,0x0F13,'AL'],[0x0F14,0x0F14,'EX'],[0x0F15,0x0F17,'AL']])
call extend(s:tmp, [[0x0F18,0x0F19,'CM'],[0x0F1A,0x0F1F,'AL'],[0x0F20,0x0F29,'NU']])
call extend(s:tmp, [[0x0F2A,0x0F33,'AL'],[0x0F34,0x0F34,'BA'],[0x0F35,0x0F35,'CM']])
call extend(s:tmp, [[0x0F36,0x0F36,'AL'],[0x0F37,0x0F37,'CM'],[0x0F38,0x0F38,'AL']])
call extend(s:tmp, [[0x0F39,0x0F39,'CM'],[0x0F3A,0x0F3A,'OP'],[0x0F3B,0x0F3B,'CL']])
call extend(s:tmp, [[0x0F3C,0x0F3C,'OP'],[0x0F3D,0x0F3D,'CL'],[0x0F3E,0x0F3F,'CM']])
call extend(s:tmp, [[0x0F40,0x0F47,'AL'],[0x0F49,0x0F6C,'AL'],[0x0F71,0x0F7E,'CM']])
call extend(s:tmp, [[0x0F7F,0x0F7F,'BA'],[0x0F80,0x0F84,'CM'],[0x0F85,0x0F85,'BA']])
call extend(s:tmp, [[0x0F86,0x0F87,'CM'],[0x0F88,0x0F8C,'AL'],[0x0F8D,0x0F97,'CM']])
call extend(s:tmp, [[0x0F99,0x0FBC,'CM'],[0x0FBE,0x0FBF,'BA'],[0x0FC0,0x0FC5,'AL']])
call extend(s:tmp, [[0x0FC6,0x0FC6,'CM'],[0x0FC7,0x0FCC,'AL'],[0x0FCE,0x0FCF,'AL']])
call extend(s:tmp, [[0x0FD0,0x0FD1,'BB'],[0x0FD2,0x0FD2,'BA'],[0x0FD3,0x0FD3,'BB']])
call extend(s:tmp, [[0x0FD4,0x0FD8,'AL'],[0x0FD9,0x0FDA,'GL'],[0x1000,0x103F,'SA']])
call extend(s:tmp, [[0x1040,0x1049,'NU'],[0x104A,0x104B,'BA'],[0x104C,0x104F,'AL']])
call extend(s:tmp, [[0x1050,0x108F,'SA'],[0x1090,0x1099,'NU'],[0x109A,0x109F,'SA']])
call extend(s:tmp, [[0x10A0,0x10C5,'AL'],[0x10C7,0x10C7,'AL'],[0x10CD,0x10CD,'AL']])
call extend(s:tmp, [[0x10D0,0x10FF,'AL'],[0x1100,0x115F,'JL'],[0x1160,0x11A7,'JV']])
call extend(s:tmp, [[0x11A8,0x11FF,'JT'],[0x1200,0x1248,'AL'],[0x124A,0x124D,'AL']])
call extend(s:tmp, [[0x1250,0x1256,'AL'],[0x1258,0x1258,'AL'],[0x125A,0x125D,'AL']])
call extend(s:tmp, [[0x1260,0x1288,'AL'],[0x128A,0x128D,'AL'],[0x1290,0x12B0,'AL']])
call extend(s:tmp, [[0x12B2,0x12B5,'AL'],[0x12B8,0x12BE,'AL'],[0x12C0,0x12C0,'AL']])
call extend(s:tmp, [[0x12C2,0x12C5,'AL'],[0x12C8,0x12D6,'AL'],[0x12D8,0x1310,'AL']])
call extend(s:tmp, [[0x1312,0x1315,'AL'],[0x1318,0x135A,'AL'],[0x135D,0x135F,'CM']])
call extend(s:tmp, [[0x1360,0x1360,'AL'],[0x1361,0x1361,'BA'],[0x1362,0x137C,'AL']])
call extend(s:tmp, [[0x1380,0x1399,'AL'],[0x13A0,0x13F5,'AL'],[0x13F8,0x13FD,'AL']])
call extend(s:tmp, [[0x1400,0x1400,'BA'],[0x1401,0x167F,'AL'],[0x1680,0x1680,'BA']])
call extend(s:tmp, [[0x1681,0x169A,'AL'],[0x169B,0x169B,'OP'],[0x169C,0x169C,'CL']])
call extend(s:tmp, [[0x16A0,0x16EA,'AL'],[0x16EB,0x16ED,'BA'],[0x16EE,0x16F8,'AL']])
call extend(s:tmp, [[0x1700,0x170C,'AL'],[0x170E,0x1711,'AL'],[0x1712,0x1714,'CM']])
call extend(s:tmp, [[0x1720,0x1731,'AL'],[0x1732,0x1734,'CM'],[0x1735,0x1736,'BA']])
call extend(s:tmp, [[0x1740,0x1751,'AL'],[0x1752,0x1753,'CM'],[0x1760,0x176C,'AL']])
call extend(s:tmp, [[0x176E,0x1770,'AL'],[0x1772,0x1773,'CM'],[0x1780,0x17D3,'SA']])
call extend(s:tmp, [[0x17D4,0x17D5,'BA'],[0x17D6,0x17D6,'NS'],[0x17D7,0x17D7,'SA']])
call extend(s:tmp, [[0x17D8,0x17D8,'BA'],[0x17D9,0x17D9,'AL'],[0x17DA,0x17DA,'BA']])
call extend(s:tmp, [[0x17DB,0x17DB,'PR'],[0x17DC,0x17DD,'SA'],[0x17E0,0x17E9,'NU']])
call extend(s:tmp, [[0x17F0,0x17F9,'AL'],[0x1800,0x1801,'AL'],[0x1802,0x1803,'EX']])
call extend(s:tmp, [[0x1804,0x1805,'BA'],[0x1806,0x1806,'BB'],[0x1807,0x1807,'AL']])
call extend(s:tmp, [[0x1808,0x1809,'EX'],[0x180A,0x180A,'AL'],[0x180B,0x180D,'CM']])
call extend(s:tmp, [[0x180E,0x180E,'GL'],[0x1810,0x1819,'NU'],[0x1820,0x1877,'AL']])
call extend(s:tmp, [[0x1880,0x1884,'AL'],[0x1885,0x1886,'CM'],[0x1887,0x18A8,'AL']])
call extend(s:tmp, [[0x18A9,0x18A9,'CM'],[0x18AA,0x18AA,'AL'],[0x18B0,0x18F5,'AL']])
call extend(s:tmp, [[0x1900,0x191E,'AL'],[0x1920,0x192B,'CM'],[0x1930,0x193B,'CM']])
call extend(s:tmp, [[0x1940,0x1940,'AL'],[0x1944,0x1945,'EX'],[0x1946,0x194F,'NU']])
call extend(s:tmp, [[0x1950,0x196D,'SA'],[0x1970,0x1974,'SA'],[0x1980,0x19AB,'SA']])
call extend(s:tmp, [[0x19B0,0x19C9,'SA'],[0x19D0,0x19D9,'NU'],[0x19DA,0x19DA,'SA']])
call extend(s:tmp, [[0x19DE,0x19DF,'SA'],[0x19E0,0x1A16,'AL'],[0x1A17,0x1A1B,'CM']])
call extend(s:tmp, [[0x1A1E,0x1A1F,'AL'],[0x1A20,0x1A5E,'SA'],[0x1A60,0x1A7C,'SA']])
call extend(s:tmp, [[0x1A7F,0x1A7F,'CM'],[0x1A80,0x1A89,'NU'],[0x1A90,0x1A99,'NU']])
call extend(s:tmp, [[0x1AA0,0x1AAD,'SA'],[0x1AB0,0x1ABE,'CM'],[0x1B00,0x1B04,'CM']])
call extend(s:tmp, [[0x1B05,0x1B33,'AL'],[0x1B34,0x1B44,'CM'],[0x1B45,0x1B4B,'AL']])
call extend(s:tmp, [[0x1B50,0x1B59,'NU'],[0x1B5A,0x1B5B,'BA'],[0x1B5C,0x1B5C,'AL']])
call extend(s:tmp, [[0x1B5D,0x1B60,'BA'],[0x1B61,0x1B6A,'AL'],[0x1B6B,0x1B73,'CM']])
call extend(s:tmp, [[0x1B74,0x1B7C,'AL'],[0x1B80,0x1B82,'CM'],[0x1B83,0x1BA0,'AL']])
call extend(s:tmp, [[0x1BA1,0x1BAD,'CM'],[0x1BAE,0x1BAF,'AL'],[0x1BB0,0x1BB9,'NU']])
call extend(s:tmp, [[0x1BBA,0x1BE5,'AL'],[0x1BE6,0x1BF3,'CM'],[0x1BFC,0x1C23,'AL']])
call extend(s:tmp, [[0x1C24,0x1C37,'CM'],[0x1C3B,0x1C3F,'BA'],[0x1C40,0x1C49,'NU']])
call extend(s:tmp, [[0x1C4D,0x1C4F,'AL'],[0x1C50,0x1C59,'NU'],[0x1C5A,0x1C7D,'AL']])
call extend(s:tmp, [[0x1C7E,0x1C7F,'BA'],[0x1C80,0x1C88,'AL'],[0x1CC0,0x1CC7,'AL']])
call extend(s:tmp, [[0x1CD0,0x1CD2,'CM'],[0x1CD3,0x1CD3,'AL'],[0x1CD4,0x1CE8,'CM']])
call extend(s:tmp, [[0x1CE9,0x1CEC,'AL'],[0x1CED,0x1CED,'CM'],[0x1CEE,0x1CF1,'AL']])
call extend(s:tmp, [[0x1CF2,0x1CF4,'CM'],[0x1CF5,0x1CF6,'AL'],[0x1CF8,0x1CF9,'CM']])
call extend(s:tmp, [[0x1D00,0x1DBF,'AL'],[0x1DC0,0x1DF5,'CM'],[0x1DFB,0x1DFF,'CM']])
call extend(s:tmp, [[0x1E00,0x1F15,'AL'],[0x1F18,0x1F1D,'AL'],[0x1F20,0x1F45,'AL']])
call extend(s:tmp, [[0x1F48,0x1F4D,'AL'],[0x1F50,0x1F57,'AL'],[0x1F59,0x1F59,'AL']])
call extend(s:tmp, [[0x1F5B,0x1F5B,'AL'],[0x1F5D,0x1F5D,'AL'],[0x1F5F,0x1F7D,'AL']])
call extend(s:tmp, [[0x1F80,0x1FB4,'AL'],[0x1FB6,0x1FC4,'AL'],[0x1FC6,0x1FD3,'AL']])
call extend(s:tmp, [[0x1FD6,0x1FDB,'AL'],[0x1FDD,0x1FEF,'AL'],[0x1FF2,0x1FF4,'AL']])
call extend(s:tmp, [[0x1FF6,0x1FFC,'AL'],[0x1FFD,0x1FFD,'BB'],[0x1FFE,0x1FFE,'AL']])
call extend(s:tmp, [[0x2000,0x2006,'BA'],[0x2007,0x2007,'GL'],[0x2008,0x200A,'BA']])
call extend(s:tmp, [[0x200B,0x200B,'ZW'],[0x200C,0x200C,'CM'],[0x200D,0x200D,'ZWJ']])
call extend(s:tmp, [[0x200E,0x200F,'CM'],[0x2010,0x2010,'BA'],[0x2011,0x2011,'GL']])
call extend(s:tmp, [[0x2012,0x2013,'BA'],[0x2014,0x2014,'B2'],[0x2015,0x2016,'AI']])
call extend(s:tmp, [[0x2017,0x2017,'AL'],[0x2018,0x2019,'QU'],[0x201A,0x201A,'OP']])
call extend(s:tmp, [[0x201B,0x201D,'QU'],[0x201E,0x201E,'OP'],[0x201F,0x201F,'QU']])
call extend(s:tmp, [[0x2020,0x2021,'AI'],[0x2022,0x2023,'AL'],[0x2024,0x2026,'IN']])
call extend(s:tmp, [[0x2027,0x2027,'BA'],[0x2028,0x2029,'BK'],[0x202A,0x202E,'CM']])
call extend(s:tmp, [[0x202F,0x202F,'GL'],[0x2030,0x2037,'PO'],[0x2038,0x2038,'AL']])
call extend(s:tmp, [[0x2039,0x203A,'QU'],[0x203B,0x203B,'AI'],[0x203C,0x203D,'NS']])
call extend(s:tmp, [[0x203E,0x2043,'AL'],[0x2044,0x2044,'IS'],[0x2045,0x2045,'OP']])
call extend(s:tmp, [[0x2046,0x2046,'CL'],[0x2047,0x2049,'NS'],[0x204A,0x2055,'AL']])
call extend(s:tmp, [[0x2056,0x2056,'BA'],[0x2057,0x2057,'AL'],[0x2058,0x205B,'BA']])
call extend(s:tmp, [[0x205C,0x205C,'AL'],[0x205D,0x205F,'BA'],[0x2060,0x2060,'WJ']])
call extend(s:tmp, [[0x2061,0x2064,'AL'],[0x2066,0x206F,'CM'],[0x2070,0x2071,'AL']])
call extend(s:tmp, [[0x2074,0x2074,'AI'],[0x2075,0x207C,'AL'],[0x207D,0x207D,'OP']])
call extend(s:tmp, [[0x207E,0x207E,'CL'],[0x207F,0x207F,'AI'],[0x2080,0x2080,'AL']])
call extend(s:tmp, [[0x2081,0x2084,'AI'],[0x2085,0x208C,'AL'],[0x208D,0x208D,'OP']])
call extend(s:tmp, [[0x208E,0x208E,'CL'],[0x2090,0x209C,'AL'],[0x20A0,0x20A6,'PR']])
call extend(s:tmp, [[0x20A7,0x20A7,'PO'],[0x20A8,0x20B5,'PR'],[0x20B6,0x20B6,'PO']])
call extend(s:tmp, [[0x20B7,0x20BA,'PR'],[0x20BB,0x20BB,'PO'],[0x20BC,0x20BD,'PR']])
call extend(s:tmp, [[0x20BE,0x20BE,'PO'],[0x20BF,0x20CF,'PR'],[0x20D0,0x20F0,'CM']])
call extend(s:tmp, [[0x2100,0x2102,'AL'],[0x2103,0x2103,'PO'],[0x2104,0x2104,'AL']])
call extend(s:tmp, [[0x2105,0x2105,'AI'],[0x2106,0x2108,'AL'],[0x2109,0x2109,'PO']])
call extend(s:tmp, [[0x210A,0x2112,'AL'],[0x2113,0x2113,'AI'],[0x2114,0x2115,'AL']])
call extend(s:tmp, [[0x2116,0x2116,'PR'],[0x2117,0x2120,'AL'],[0x2121,0x2122,'AI']])
call extend(s:tmp, [[0x2123,0x212A,'AL'],[0x212B,0x212B,'AI'],[0x212C,0x2153,'AL']])
call extend(s:tmp, [[0x2154,0x2155,'AI'],[0x2156,0x215A,'AL'],[0x215B,0x215B,'AI']])
call extend(s:tmp, [[0x215C,0x215D,'AL'],[0x215E,0x215E,'AI'],[0x215F,0x215F,'AL']])
call extend(s:tmp, [[0x2160,0x216B,'AI'],[0x216C,0x216F,'AL'],[0x2170,0x2179,'AI']])
call extend(s:tmp, [[0x217A,0x2188,'AL'],[0x2189,0x2189,'AI'],[0x218A,0x218B,'AL']])
call extend(s:tmp, [[0x2190,0x2199,'AI'],[0x219A,0x21D1,'AL'],[0x21D2,0x21D2,'AI']])
call extend(s:tmp, [[0x21D3,0x21D3,'AL'],[0x21D4,0x21D4,'AI'],[0x21D5,0x21FF,'AL']])
call extend(s:tmp, [[0x2200,0x2200,'AI'],[0x2201,0x2201,'AL'],[0x2202,0x2203,'AI']])
call extend(s:tmp, [[0x2204,0x2206,'AL'],[0x2207,0x2208,'AI'],[0x2209,0x220A,'AL']])
call extend(s:tmp, [[0x220B,0x220B,'AI'],[0x220C,0x220E,'AL'],[0x220F,0x220F,'AI']])
call extend(s:tmp, [[0x2210,0x2210,'AL'],[0x2211,0x2211,'AI'],[0x2212,0x2213,'PR']])
call extend(s:tmp, [[0x2214,0x2214,'AL'],[0x2215,0x2215,'AI'],[0x2216,0x2219,'AL']])
call extend(s:tmp, [[0x221A,0x221A,'AI'],[0x221B,0x221C,'AL'],[0x221D,0x2220,'AI']])
call extend(s:tmp, [[0x2221,0x2222,'AL'],[0x2223,0x2223,'AI'],[0x2224,0x2224,'AL']])
call extend(s:tmp, [[0x2225,0x2225,'AI'],[0x2226,0x2226,'AL'],[0x2227,0x222C,'AI']])
call extend(s:tmp, [[0x222D,0x222D,'AL'],[0x222E,0x222E,'AI'],[0x222F,0x2233,'AL']])
call extend(s:tmp, [[0x2234,0x2237,'AI'],[0x2238,0x223B,'AL'],[0x223C,0x223D,'AI']])
call extend(s:tmp, [[0x223E,0x2247,'AL'],[0x2248,0x2248,'AI'],[0x2249,0x224B,'AL']])
call extend(s:tmp, [[0x224C,0x224C,'AI'],[0x224D,0x2251,'AL'],[0x2252,0x2252,'AI']])
call extend(s:tmp, [[0x2253,0x225F,'AL'],[0x2260,0x2261,'AI'],[0x2262,0x2263,'AL']])
call extend(s:tmp, [[0x2264,0x2267,'AI'],[0x2268,0x2269,'AL'],[0x226A,0x226B,'AI']])
call extend(s:tmp, [[0x226C,0x226D,'AL'],[0x226E,0x226F,'AI'],[0x2270,0x2281,'AL']])
call extend(s:tmp, [[0x2282,0x2283,'AI'],[0x2284,0x2285,'AL'],[0x2286,0x2287,'AI']])
call extend(s:tmp, [[0x2288,0x2294,'AL'],[0x2295,0x2295,'AI'],[0x2296,0x2298,'AL']])
call extend(s:tmp, [[0x2299,0x2299,'AI'],[0x229A,0x22A4,'AL'],[0x22A5,0x22A5,'AI']])
call extend(s:tmp, [[0x22A6,0x22BE,'AL'],[0x22BF,0x22BF,'AI'],[0x22C0,0x22EE,'AL']])
call extend(s:tmp, [[0x22EF,0x22EF,'IN'],[0x22F0,0x2307,'AL'],[0x2308,0x2308,'OP']])
call extend(s:tmp, [[0x2309,0x2309,'CL'],[0x230A,0x230A,'OP'],[0x230B,0x230B,'CL']])
call extend(s:tmp, [[0x230C,0x2311,'AL'],[0x2312,0x2312,'AI'],[0x2313,0x2319,'AL']])
call extend(s:tmp, [[0x231A,0x231B,'ID'],[0x231C,0x2328,'AL'],[0x2329,0x2329,'OP']])
call extend(s:tmp, [[0x232A,0x232A,'CL'],[0x232B,0x23EF,'AL'],[0x23F0,0x23F3,'ID']])
call extend(s:tmp, [[0x23F4,0x23FE,'AL'],[0x2400,0x2426,'AL'],[0x2440,0x244A,'AL']])
call extend(s:tmp, [[0x2460,0x24FE,'AI'],[0x24FF,0x24FF,'AL'],[0x2500,0x254B,'AI']])
call extend(s:tmp, [[0x254C,0x254F,'AL'],[0x2550,0x2574,'AI'],[0x2575,0x257F,'AL']])
call extend(s:tmp, [[0x2580,0x258F,'AI'],[0x2590,0x2591,'AL'],[0x2592,0x2595,'AI']])
call extend(s:tmp, [[0x2596,0x259F,'AL'],[0x25A0,0x25A1,'AI'],[0x25A2,0x25A2,'AL']])
call extend(s:tmp, [[0x25A3,0x25A9,'AI'],[0x25AA,0x25B1,'AL'],[0x25B2,0x25B3,'AI']])
call extend(s:tmp, [[0x25B4,0x25B5,'AL'],[0x25B6,0x25B7,'AI'],[0x25B8,0x25BB,'AL']])
call extend(s:tmp, [[0x25BC,0x25BD,'AI'],[0x25BE,0x25BF,'AL'],[0x25C0,0x25C1,'AI']])
call extend(s:tmp, [[0x25C2,0x25C5,'AL'],[0x25C6,0x25C8,'AI'],[0x25C9,0x25CA,'AL']])
call extend(s:tmp, [[0x25CB,0x25CB,'AI'],[0x25CC,0x25CD,'AL'],[0x25CE,0x25D1,'AI']])
call extend(s:tmp, [[0x25D2,0x25E1,'AL'],[0x25E2,0x25E5,'AI'],[0x25E6,0x25EE,'AL']])
call extend(s:tmp, [[0x25EF,0x25EF,'AI'],[0x25F0,0x25FF,'AL'],[0x2600,0x2603,'ID']])
call extend(s:tmp, [[0x2604,0x2604,'AL'],[0x2605,0x2606,'AI'],[0x2607,0x2608,'AL']])
call extend(s:tmp, [[0x2609,0x2609,'AI'],[0x260A,0x260D,'AL'],[0x260E,0x260F,'AI']])
call extend(s:tmp, [[0x2610,0x2613,'AL'],[0x2614,0x2615,'ID'],[0x2616,0x2617,'AI']])
call extend(s:tmp, [[0x2618,0x2618,'ID'],[0x2619,0x2619,'AL'],[0x261A,0x261C,'ID']])
call extend(s:tmp, [[0x261D,0x261D,'EB'],[0x261E,0x261F,'ID'],[0x2620,0x2638,'AL']])
call extend(s:tmp, [[0x2639,0x263B,'ID'],[0x263C,0x263F,'AL'],[0x2640,0x2640,'AI']])
call extend(s:tmp, [[0x2641,0x2641,'AL'],[0x2642,0x2642,'AI'],[0x2643,0x265F,'AL']])
call extend(s:tmp, [[0x2660,0x2661,'AI'],[0x2662,0x2662,'AL'],[0x2663,0x2665,'AI']])
call extend(s:tmp, [[0x2666,0x2666,'AL'],[0x2667,0x2667,'AI'],[0x2668,0x2668,'ID']])
call extend(s:tmp, [[0x2669,0x266A,'AI'],[0x266B,0x266B,'AL'],[0x266C,0x266D,'AI']])
call extend(s:tmp, [[0x266E,0x266E,'AL'],[0x266F,0x266F,'AI'],[0x2670,0x267E,'AL']])
call extend(s:tmp, [[0x267F,0x267F,'ID'],[0x2680,0x269D,'AL'],[0x269E,0x269F,'AI']])
call extend(s:tmp, [[0x26A0,0x26BC,'AL'],[0x26BD,0x26C8,'ID'],[0x26C9,0x26CC,'AI']])
call extend(s:tmp, [[0x26CD,0x26CD,'ID'],[0x26CE,0x26CE,'AL'],[0x26CF,0x26D1,'ID']])
call extend(s:tmp, [[0x26D2,0x26D2,'AI'],[0x26D3,0x26D4,'ID'],[0x26D5,0x26D7,'AI']])
call extend(s:tmp, [[0x26D8,0x26D9,'ID'],[0x26DA,0x26DB,'AI'],[0x26DC,0x26DC,'ID']])
call extend(s:tmp, [[0x26DD,0x26DE,'AI'],[0x26DF,0x26E1,'ID'],[0x26E2,0x26E2,'AL']])
call extend(s:tmp, [[0x26E3,0x26E3,'AI'],[0x26E4,0x26E7,'AL'],[0x26E8,0x26E9,'AI']])
call extend(s:tmp, [[0x26EA,0x26EA,'ID'],[0x26EB,0x26F0,'AI'],[0x26F1,0x26F5,'ID']])
call extend(s:tmp, [[0x26F6,0x26F6,'AI'],[0x26F7,0x26F8,'ID'],[0x26F9,0x26F9,'EB']])
call extend(s:tmp, [[0x26FA,0x26FA,'ID'],[0x26FB,0x26FC,'AI'],[0x26FD,0x2704,'ID']])
call extend(s:tmp, [[0x2705,0x2707,'AL'],[0x2708,0x2709,'ID'],[0x270A,0x270D,'EB']])
call extend(s:tmp, [[0x270E,0x2756,'AL'],[0x2757,0x2757,'AI'],[0x2758,0x275A,'AL']])
call extend(s:tmp, [[0x275B,0x2760,'QU'],[0x2761,0x2761,'AL'],[0x2762,0x2763,'EX']])
call extend(s:tmp, [[0x2764,0x2764,'ID'],[0x2765,0x2767,'AL'],[0x2768,0x2768,'OP']])
call extend(s:tmp, [[0x2769,0x2769,'CL'],[0x276A,0x276A,'OP'],[0x276B,0x276B,'CL']])
call extend(s:tmp, [[0x276C,0x276C,'OP'],[0x276D,0x276D,'CL'],[0x276E,0x276E,'OP']])
call extend(s:tmp, [[0x276F,0x276F,'CL'],[0x2770,0x2770,'OP'],[0x2771,0x2771,'CL']])
call extend(s:tmp, [[0x2772,0x2772,'OP'],[0x2773,0x2773,'CL'],[0x2774,0x2774,'OP']])
call extend(s:tmp, [[0x2775,0x2775,'CL'],[0x2776,0x2793,'AI'],[0x2794,0x27C4,'AL']])
call extend(s:tmp, [[0x27C5,0x27C5,'OP'],[0x27C6,0x27C6,'CL'],[0x27C7,0x27E5,'AL']])
call extend(s:tmp, [[0x27E6,0x27E6,'OP'],[0x27E7,0x27E7,'CL'],[0x27E8,0x27E8,'OP']])
call extend(s:tmp, [[0x27E9,0x27E9,'CL'],[0x27EA,0x27EA,'OP'],[0x27EB,0x27EB,'CL']])
call extend(s:tmp, [[0x27EC,0x27EC,'OP'],[0x27ED,0x27ED,'CL'],[0x27EE,0x27EE,'OP']])
call extend(s:tmp, [[0x27EF,0x27EF,'CL'],[0x27F0,0x2982,'AL'],[0x2983,0x2983,'OP']])
call extend(s:tmp, [[0x2984,0x2984,'CL'],[0x2985,0x2985,'OP'],[0x2986,0x2986,'CL']])
call extend(s:tmp, [[0x2987,0x2987,'OP'],[0x2988,0x2988,'CL'],[0x2989,0x2989,'OP']])
call extend(s:tmp, [[0x298A,0x298A,'CL'],[0x298B,0x298B,'OP'],[0x298C,0x298C,'CL']])
call extend(s:tmp, [[0x298D,0x298D,'OP'],[0x298E,0x298E,'CL'],[0x298F,0x298F,'OP']])
call extend(s:tmp, [[0x2990,0x2990,'CL'],[0x2991,0x2991,'OP'],[0x2992,0x2992,'CL']])
call extend(s:tmp, [[0x2993,0x2993,'OP'],[0x2994,0x2994,'CL'],[0x2995,0x2995,'OP']])
call extend(s:tmp, [[0x2996,0x2996,'CL'],[0x2997,0x2997,'OP'],[0x2998,0x2998,'CL']])
call extend(s:tmp, [[0x2999,0x29D7,'AL'],[0x29D8,0x29D8,'OP'],[0x29D9,0x29D9,'CL']])
call extend(s:tmp, [[0x29DA,0x29DA,'OP'],[0x29DB,0x29DB,'CL'],[0x29DC,0x29FB,'AL']])
call extend(s:tmp, [[0x29FC,0x29FC,'OP'],[0x29FD,0x29FD,'CL'],[0x29FE,0x2B54,'AL']])
call extend(s:tmp, [[0x2B55,0x2B59,'AI'],[0x2B5A,0x2B73,'AL'],[0x2B76,0x2B95,'AL']])
call extend(s:tmp, [[0x2B98,0x2BB9,'AL'],[0x2BBD,0x2BC8,'AL'],[0x2BCA,0x2BD1,'AL']])
call extend(s:tmp, [[0x2BEC,0x2BEF,'AL'],[0x2C00,0x2C2E,'AL'],[0x2C30,0x2C5E,'AL']])
call extend(s:tmp, [[0x2C60,0x2CEE,'AL'],[0x2CEF,0x2CF1,'CM'],[0x2CF2,0x2CF3,'AL']])
call extend(s:tmp, [[0x2CF9,0x2CF9,'EX'],[0x2CFA,0x2CFC,'BA'],[0x2CFD,0x2CFD,'AL']])
call extend(s:tmp, [[0x2CFE,0x2CFE,'EX'],[0x2CFF,0x2CFF,'BA'],[0x2D00,0x2D25,'AL']])
call extend(s:tmp, [[0x2D27,0x2D27,'AL'],[0x2D2D,0x2D2D,'AL'],[0x2D30,0x2D67,'AL']])
call extend(s:tmp, [[0x2D6F,0x2D6F,'AL'],[0x2D70,0x2D70,'BA'],[0x2D7F,0x2D7F,'CM']])
call extend(s:tmp, [[0x2D80,0x2D96,'AL'],[0x2DA0,0x2DA6,'AL'],[0x2DA8,0x2DAE,'AL']])
call extend(s:tmp, [[0x2DB0,0x2DB6,'AL'],[0x2DB8,0x2DBE,'AL'],[0x2DC0,0x2DC6,'AL']])
call extend(s:tmp, [[0x2DC8,0x2DCE,'AL'],[0x2DD0,0x2DD6,'AL'],[0x2DD8,0x2DDE,'AL']])
call extend(s:tmp, [[0x2DE0,0x2DFF,'CM'],[0x2E00,0x2E0D,'QU'],[0x2E0E,0x2E15,'BA']])
call extend(s:tmp, [[0x2E16,0x2E16,'AL'],[0x2E17,0x2E17,'BA'],[0x2E18,0x2E18,'OP']])
call extend(s:tmp, [[0x2E19,0x2E19,'BA'],[0x2E1A,0x2E1B,'AL'],[0x2E1C,0x2E1D,'QU']])
call extend(s:tmp, [[0x2E1E,0x2E1F,'AL'],[0x2E20,0x2E21,'QU'],[0x2E22,0x2E22,'OP']])
call extend(s:tmp, [[0x2E23,0x2E23,'CL'],[0x2E24,0x2E24,'OP'],[0x2E25,0x2E25,'CL']])
call extend(s:tmp, [[0x2E26,0x2E26,'OP'],[0x2E27,0x2E27,'CL'],[0x2E28,0x2E28,'OP']])
call extend(s:tmp, [[0x2E29,0x2E29,'CL'],[0x2E2A,0x2E2D,'BA'],[0x2E2E,0x2E2E,'EX']])
call extend(s:tmp, [[0x2E2F,0x2E2F,'AL'],[0x2E30,0x2E31,'BA'],[0x2E32,0x2E32,'AL']])
call extend(s:tmp, [[0x2E33,0x2E34,'BA'],[0x2E35,0x2E39,'AL'],[0x2E3A,0x2E3B,'B2']])
call extend(s:tmp, [[0x2E3C,0x2E3E,'BA'],[0x2E3F,0x2E3F,'AL'],[0x2E40,0x2E41,'BA']])
call extend(s:tmp, [[0x2E42,0x2E42,'OP'],[0x2E43,0x2E44,'BA'],[0x2E80,0x2E99,'ID']])
call extend(s:tmp, [[0x2E9B,0x2EF3,'ID'],[0x2F00,0x2FD5,'ID'],[0x2FF0,0x2FFB,'ID']])
call extend(s:tmp, [[0x3000,0x3000,'BA'],[0x3001,0x3002,'CL'],[0x3003,0x3004,'ID']])
call extend(s:tmp, [[0x3005,0x3005,'NS'],[0x3006,0x3007,'ID'],[0x3008,0x3008,'OP']])
call extend(s:tmp, [[0x3009,0x3009,'CL'],[0x300A,0x300A,'OP'],[0x300B,0x300B,'CL']])
call extend(s:tmp, [[0x300C,0x300C,'OP'],[0x300D,0x300D,'CL'],[0x300E,0x300E,'OP']])
call extend(s:tmp, [[0x300F,0x300F,'CL'],[0x3010,0x3010,'OP'],[0x3011,0x3011,'CL']])
call extend(s:tmp, [[0x3012,0x3013,'ID'],[0x3014,0x3014,'OP'],[0x3015,0x3015,'CL']])
call extend(s:tmp, [[0x3016,0x3016,'OP'],[0x3017,0x3017,'CL'],[0x3018,0x3018,'OP']])
call extend(s:tmp, [[0x3019,0x3019,'CL'],[0x301A,0x301A,'OP'],[0x301B,0x301B,'CL']])
call extend(s:tmp, [[0x301C,0x301C,'NS'],[0x301D,0x301D,'OP'],[0x301E,0x301F,'CL']])
call extend(s:tmp, [[0x3020,0x3029,'ID'],[0x302A,0x302F,'CM'],[0x3030,0x3034,'ID']])
call extend(s:tmp, [[0x3035,0x3035,'CM'],[0x3036,0x303A,'ID'],[0x303B,0x303C,'NS']])
call extend(s:tmp, [[0x303D,0x303F,'ID'],[0x3041,0x3041,'CJ'],[0x3042,0x3042,'ID']])
call extend(s:tmp, [[0x3043,0x3043,'CJ'],[0x3044,0x3044,'ID'],[0x3045,0x3045,'CJ']])
call extend(s:tmp, [[0x3046,0x3046,'ID'],[0x3047,0x3047,'CJ'],[0x3048,0x3048,'ID']])
call extend(s:tmp, [[0x3049,0x3049,'CJ'],[0x304A,0x3062,'ID'],[0x3063,0x3063,'CJ']])
call extend(s:tmp, [[0x3064,0x3082,'ID'],[0x3083,0x3083,'CJ'],[0x3084,0x3084,'ID']])
call extend(s:tmp, [[0x3085,0x3085,'CJ'],[0x3086,0x3086,'ID'],[0x3087,0x3087,'CJ']])
call extend(s:tmp, [[0x3088,0x308D,'ID'],[0x308E,0x308E,'CJ'],[0x308F,0x3094,'ID']])
call extend(s:tmp, [[0x3095,0x3096,'CJ'],[0x3099,0x309A,'CM'],[0x309B,0x309E,'NS']])
call extend(s:tmp, [[0x309F,0x309F,'ID'],[0x30A0,0x30A0,'NS'],[0x30A1,0x30A1,'CJ']])
call extend(s:tmp, [[0x30A2,0x30A2,'ID'],[0x30A3,0x30A3,'CJ'],[0x30A4,0x30A4,'ID']])
call extend(s:tmp, [[0x30A5,0x30A5,'CJ'],[0x30A6,0x30A6,'ID'],[0x30A7,0x30A7,'CJ']])
call extend(s:tmp, [[0x30A8,0x30A8,'ID'],[0x30A9,0x30A9,'CJ'],[0x30AA,0x30C2,'ID']])
call extend(s:tmp, [[0x30C3,0x30C3,'CJ'],[0x30C4,0x30E2,'ID'],[0x30E3,0x30E3,'CJ']])
call extend(s:tmp, [[0x30E4,0x30E4,'ID'],[0x30E5,0x30E5,'CJ'],[0x30E6,0x30E6,'ID']])
call extend(s:tmp, [[0x30E7,0x30E7,'CJ'],[0x30E8,0x30ED,'ID'],[0x30EE,0x30EE,'CJ']])
call extend(s:tmp, [[0x30EF,0x30F4,'ID'],[0x30F5,0x30F6,'CJ'],[0x30F7,0x30FA,'ID']])
call extend(s:tmp, [[0x30FB,0x30FB,'NS'],[0x30FC,0x30FC,'CJ'],[0x30FD,0x30FE,'NS']])
call extend(s:tmp, [[0x30FF,0x30FF,'ID'],[0x3105,0x312D,'ID'],[0x3131,0x318E,'ID']])
call extend(s:tmp, [[0x3190,0x31BA,'ID'],[0x31C0,0x31E3,'ID'],[0x31F0,0x31FF,'CJ']])
call extend(s:tmp, [[0x3200,0x321E,'ID'],[0x3220,0x3247,'ID'],[0x3248,0x324F,'AI']])
call extend(s:tmp, [[0x3250,0x32FE,'ID'],[0x3300,0x4DBF,'ID'],[0x4DC0,0x4DFF,'AL']])
call extend(s:tmp, [[0x4E00,0xA014,'ID'],[0xA015,0xA015,'NS'],[0xA016,0xA48C,'ID']])
call extend(s:tmp, [[0xA490,0xA4C6,'ID'],[0xA4D0,0xA4FD,'AL'],[0xA4FE,0xA4FF,'BA']])
call extend(s:tmp, [[0xA500,0xA60C,'AL'],[0xA60D,0xA60D,'BA'],[0xA60E,0xA60E,'EX']])
call extend(s:tmp, [[0xA60F,0xA60F,'BA'],[0xA610,0xA61F,'AL'],[0xA620,0xA629,'NU']])
call extend(s:tmp, [[0xA62A,0xA62B,'AL'],[0xA640,0xA66E,'AL'],[0xA66F,0xA672,'CM']])
call extend(s:tmp, [[0xA673,0xA673,'AL'],[0xA674,0xA67D,'CM'],[0xA67E,0xA69D,'AL']])
call extend(s:tmp, [[0xA69E,0xA69F,'CM'],[0xA6A0,0xA6EF,'AL'],[0xA6F0,0xA6F1,'CM']])
call extend(s:tmp, [[0xA6F2,0xA6F2,'AL'],[0xA6F3,0xA6F7,'BA'],[0xA700,0xA7AE,'AL']])
call extend(s:tmp, [[0xA7B0,0xA7B7,'AL'],[0xA7F7,0xA801,'AL'],[0xA802,0xA802,'CM']])
call extend(s:tmp, [[0xA803,0xA805,'AL'],[0xA806,0xA806,'CM'],[0xA807,0xA80A,'AL']])
call extend(s:tmp, [[0xA80B,0xA80B,'CM'],[0xA80C,0xA822,'AL'],[0xA823,0xA827,'CM']])
call extend(s:tmp, [[0xA828,0xA82B,'AL'],[0xA830,0xA837,'AL'],[0xA838,0xA838,'PO']])
call extend(s:tmp, [[0xA839,0xA839,'AL'],[0xA840,0xA873,'AL'],[0xA874,0xA875,'BB']])
call extend(s:tmp, [[0xA876,0xA877,'EX'],[0xA880,0xA881,'CM'],[0xA882,0xA8B3,'AL']])
call extend(s:tmp, [[0xA8B4,0xA8C5,'CM'],[0xA8CE,0xA8CF,'BA'],[0xA8D0,0xA8D9,'NU']])
call extend(s:tmp, [[0xA8E0,0xA8F1,'CM'],[0xA8F2,0xA8FB,'AL'],[0xA8FC,0xA8FC,'BB']])
call extend(s:tmp, [[0xA8FD,0xA8FD,'AL'],[0xA900,0xA909,'NU'],[0xA90A,0xA925,'AL']])
call extend(s:tmp, [[0xA926,0xA92D,'CM'],[0xA92E,0xA92F,'BA'],[0xA930,0xA946,'AL']])
call extend(s:tmp, [[0xA947,0xA953,'CM'],[0xA95F,0xA95F,'AL'],[0xA960,0xA97C,'JL']])
call extend(s:tmp, [[0xA980,0xA983,'CM'],[0xA984,0xA9B2,'AL'],[0xA9B3,0xA9C0,'CM']])
call extend(s:tmp, [[0xA9C1,0xA9C6,'AL'],[0xA9C7,0xA9C9,'BA'],[0xA9CA,0xA9CD,'AL']])
call extend(s:tmp, [[0xA9CF,0xA9CF,'AL'],[0xA9D0,0xA9D9,'NU'],[0xA9DE,0xA9DF,'AL']])
call extend(s:tmp, [[0xA9E0,0xA9EF,'SA'],[0xA9F0,0xA9F9,'NU'],[0xA9FA,0xA9FE,'SA']])
call extend(s:tmp, [[0xAA00,0xAA28,'AL'],[0xAA29,0xAA36,'CM'],[0xAA40,0xAA42,'AL']])
call extend(s:tmp, [[0xAA43,0xAA43,'CM'],[0xAA44,0xAA4B,'AL'],[0xAA4C,0xAA4D,'CM']])
call extend(s:tmp, [[0xAA50,0xAA59,'NU'],[0xAA5C,0xAA5C,'AL'],[0xAA5D,0xAA5F,'BA']])
call extend(s:tmp, [[0xAA60,0xAAC2,'SA'],[0xAADB,0xAADF,'SA'],[0xAAE0,0xAAEA,'AL']])
call extend(s:tmp, [[0xAAEB,0xAAEF,'CM'],[0xAAF0,0xAAF1,'BA'],[0xAAF2,0xAAF4,'AL']])
call extend(s:tmp, [[0xAAF5,0xAAF6,'CM'],[0xAB01,0xAB06,'AL'],[0xAB09,0xAB0E,'AL']])
call extend(s:tmp, [[0xAB11,0xAB16,'AL'],[0xAB20,0xAB26,'AL'],[0xAB28,0xAB2E,'AL']])
call extend(s:tmp, [[0xAB30,0xAB65,'AL'],[0xAB70,0xABE2,'AL'],[0xABE3,0xABEA,'CM']])
call extend(s:tmp, [[0xABEB,0xABEB,'BA'],[0xABEC,0xABED,'CM'],[0xABF0,0xABF9,'NU']])
call extend(s:tmp, [[0xAC00,0xAC00,'H2'],[0xAC01,0xAC1B,'H3'],[0xAC1C,0xAC1C,'H2']])
call extend(s:tmp, [[0xAC1D,0xAC37,'H3'],[0xAC38,0xAC38,'H2'],[0xAC39,0xAC53,'H3']])
call extend(s:tmp, [[0xAC54,0xAC54,'H2'],[0xAC55,0xAC6F,'H3'],[0xAC70,0xAC70,'H2']])
call extend(s:tmp, [[0xAC71,0xAC8B,'H3'],[0xAC8C,0xAC8C,'H2'],[0xAC8D,0xACA7,'H3']])
call extend(s:tmp, [[0xACA8,0xACA8,'H2'],[0xACA9,0xACC3,'H3'],[0xACC4,0xACC4,'H2']])
call extend(s:tmp, [[0xACC5,0xACDF,'H3'],[0xACE0,0xACE0,'H2'],[0xACE1,0xACFB,'H3']])
call extend(s:tmp, [[0xACFC,0xACFC,'H2'],[0xACFD,0xAD17,'H3'],[0xAD18,0xAD18,'H2']])
call extend(s:tmp, [[0xAD19,0xAD33,'H3'],[0xAD34,0xAD34,'H2'],[0xAD35,0xAD4F,'H3']])
call extend(s:tmp, [[0xAD50,0xAD50,'H2'],[0xAD51,0xAD6B,'H3'],[0xAD6C,0xAD6C,'H2']])
call extend(s:tmp, [[0xAD6D,0xAD87,'H3'],[0xAD88,0xAD88,'H2'],[0xAD89,0xADA3,'H3']])
call extend(s:tmp, [[0xADA4,0xADA4,'H2'],[0xADA5,0xADBF,'H3'],[0xADC0,0xADC0,'H2']])
call extend(s:tmp, [[0xADC1,0xADDB,'H3'],[0xADDC,0xADDC,'H2'],[0xADDD,0xADF7,'H3']])
call extend(s:tmp, [[0xADF8,0xADF8,'H2'],[0xADF9,0xAE13,'H3'],[0xAE14,0xAE14,'H2']])
call extend(s:tmp, [[0xAE15,0xAE2F,'H3'],[0xAE30,0xAE30,'H2'],[0xAE31,0xAE4B,'H3']])
call extend(s:tmp, [[0xAE4C,0xAE4C,'H2'],[0xAE4D,0xAE67,'H3'],[0xAE68,0xAE68,'H2']])
call extend(s:tmp, [[0xAE69,0xAE83,'H3'],[0xAE84,0xAE84,'H2'],[0xAE85,0xAE9F,'H3']])
call extend(s:tmp, [[0xAEA0,0xAEA0,'H2'],[0xAEA1,0xAEBB,'H3'],[0xAEBC,0xAEBC,'H2']])
call extend(s:tmp, [[0xAEBD,0xAED7,'H3'],[0xAED8,0xAED8,'H2'],[0xAED9,0xAEF3,'H3']])
call extend(s:tmp, [[0xAEF4,0xAEF4,'H2'],[0xAEF5,0xAF0F,'H3'],[0xAF10,0xAF10,'H2']])
call extend(s:tmp, [[0xAF11,0xAF2B,'H3'],[0xAF2C,0xAF2C,'H2'],[0xAF2D,0xAF47,'H3']])
call extend(s:tmp, [[0xAF48,0xAF48,'H2'],[0xAF49,0xAF63,'H3'],[0xAF64,0xAF64,'H2']])
call extend(s:tmp, [[0xAF65,0xAF7F,'H3'],[0xAF80,0xAF80,'H2'],[0xAF81,0xAF9B,'H3']])
call extend(s:tmp, [[0xAF9C,0xAF9C,'H2'],[0xAF9D,0xAFB7,'H3'],[0xAFB8,0xAFB8,'H2']])
call extend(s:tmp, [[0xAFB9,0xAFD3,'H3'],[0xAFD4,0xAFD4,'H2'],[0xAFD5,0xAFEF,'H3']])
call extend(s:tmp, [[0xAFF0,0xAFF0,'H2'],[0xAFF1,0xB00B,'H3'],[0xB00C,0xB00C,'H2']])
call extend(s:tmp, [[0xB00D,0xB027,'H3'],[0xB028,0xB028,'H2'],[0xB029,0xB043,'H3']])
call extend(s:tmp, [[0xB044,0xB044,'H2'],[0xB045,0xB05F,'H3'],[0xB060,0xB060,'H2']])
call extend(s:tmp, [[0xB061,0xB07B,'H3'],[0xB07C,0xB07C,'H2'],[0xB07D,0xB097,'H3']])
call extend(s:tmp, [[0xB098,0xB098,'H2'],[0xB099,0xB0B3,'H3'],[0xB0B4,0xB0B4,'H2']])
call extend(s:tmp, [[0xB0B5,0xB0CF,'H3'],[0xB0D0,0xB0D0,'H2'],[0xB0D1,0xB0EB,'H3']])
call extend(s:tmp, [[0xB0EC,0xB0EC,'H2'],[0xB0ED,0xB107,'H3'],[0xB108,0xB108,'H2']])
call extend(s:tmp, [[0xB109,0xB123,'H3'],[0xB124,0xB124,'H2'],[0xB125,0xB13F,'H3']])
call extend(s:tmp, [[0xB140,0xB140,'H2'],[0xB141,0xB15B,'H3'],[0xB15C,0xB15C,'H2']])
call extend(s:tmp, [[0xB15D,0xB177,'H3'],[0xB178,0xB178,'H2'],[0xB179,0xB193,'H3']])
call extend(s:tmp, [[0xB194,0xB194,'H2'],[0xB195,0xB1AF,'H3'],[0xB1B0,0xB1B0,'H2']])
call extend(s:tmp, [[0xB1B1,0xB1CB,'H3'],[0xB1CC,0xB1CC,'H2'],[0xB1CD,0xB1E7,'H3']])
call extend(s:tmp, [[0xB1E8,0xB1E8,'H2'],[0xB1E9,0xB203,'H3'],[0xB204,0xB204,'H2']])
call extend(s:tmp, [[0xB205,0xB21F,'H3'],[0xB220,0xB220,'H2'],[0xB221,0xB23B,'H3']])
call extend(s:tmp, [[0xB23C,0xB23C,'H2'],[0xB23D,0xB257,'H3'],[0xB258,0xB258,'H2']])
call extend(s:tmp, [[0xB259,0xB273,'H3'],[0xB274,0xB274,'H2'],[0xB275,0xB28F,'H3']])
call extend(s:tmp, [[0xB290,0xB290,'H2'],[0xB291,0xB2AB,'H3'],[0xB2AC,0xB2AC,'H2']])
call extend(s:tmp, [[0xB2AD,0xB2C7,'H3'],[0xB2C8,0xB2C8,'H2'],[0xB2C9,0xB2E3,'H3']])
call extend(s:tmp, [[0xB2E4,0xB2E4,'H2'],[0xB2E5,0xB2FF,'H3'],[0xB300,0xB300,'H2']])
call extend(s:tmp, [[0xB301,0xB31B,'H3'],[0xB31C,0xB31C,'H2'],[0xB31D,0xB337,'H3']])
call extend(s:tmp, [[0xB338,0xB338,'H2'],[0xB339,0xB353,'H3'],[0xB354,0xB354,'H2']])
call extend(s:tmp, [[0xB355,0xB36F,'H3'],[0xB370,0xB370,'H2'],[0xB371,0xB38B,'H3']])
call extend(s:tmp, [[0xB38C,0xB38C,'H2'],[0xB38D,0xB3A7,'H3'],[0xB3A8,0xB3A8,'H2']])
call extend(s:tmp, [[0xB3A9,0xB3C3,'H3'],[0xB3C4,0xB3C4,'H2'],[0xB3C5,0xB3DF,'H3']])
call extend(s:tmp, [[0xB3E0,0xB3E0,'H2'],[0xB3E1,0xB3FB,'H3'],[0xB3FC,0xB3FC,'H2']])
call extend(s:tmp, [[0xB3FD,0xB417,'H3'],[0xB418,0xB418,'H2'],[0xB419,0xB433,'H3']])
call extend(s:tmp, [[0xB434,0xB434,'H2'],[0xB435,0xB44F,'H3'],[0xB450,0xB450,'H2']])
call extend(s:tmp, [[0xB451,0xB46B,'H3'],[0xB46C,0xB46C,'H2'],[0xB46D,0xB487,'H3']])
call extend(s:tmp, [[0xB488,0xB488,'H2'],[0xB489,0xB4A3,'H3'],[0xB4A4,0xB4A4,'H2']])
call extend(s:tmp, [[0xB4A5,0xB4BF,'H3'],[0xB4C0,0xB4C0,'H2'],[0xB4C1,0xB4DB,'H3']])
call extend(s:tmp, [[0xB4DC,0xB4DC,'H2'],[0xB4DD,0xB4F7,'H3'],[0xB4F8,0xB4F8,'H2']])
call extend(s:tmp, [[0xB4F9,0xB513,'H3'],[0xB514,0xB514,'H2'],[0xB515,0xB52F,'H3']])
call extend(s:tmp, [[0xB530,0xB530,'H2'],[0xB531,0xB54B,'H3'],[0xB54C,0xB54C,'H2']])
call extend(s:tmp, [[0xB54D,0xB567,'H3'],[0xB568,0xB568,'H2'],[0xB569,0xB583,'H3']])
call extend(s:tmp, [[0xB584,0xB584,'H2'],[0xB585,0xB59F,'H3'],[0xB5A0,0xB5A0,'H2']])
call extend(s:tmp, [[0xB5A1,0xB5BB,'H3'],[0xB5BC,0xB5BC,'H2'],[0xB5BD,0xB5D7,'H3']])
call extend(s:tmp, [[0xB5D8,0xB5D8,'H2'],[0xB5D9,0xB5F3,'H3'],[0xB5F4,0xB5F4,'H2']])
call extend(s:tmp, [[0xB5F5,0xB60F,'H3'],[0xB610,0xB610,'H2'],[0xB611,0xB62B,'H3']])
call extend(s:tmp, [[0xB62C,0xB62C,'H2'],[0xB62D,0xB647,'H3'],[0xB648,0xB648,'H2']])
call extend(s:tmp, [[0xB649,0xB663,'H3'],[0xB664,0xB664,'H2'],[0xB665,0xB67F,'H3']])
call extend(s:tmp, [[0xB680,0xB680,'H2'],[0xB681,0xB69B,'H3'],[0xB69C,0xB69C,'H2']])
call extend(s:tmp, [[0xB69D,0xB6B7,'H3'],[0xB6B8,0xB6B8,'H2'],[0xB6B9,0xB6D3,'H3']])
call extend(s:tmp, [[0xB6D4,0xB6D4,'H2'],[0xB6D5,0xB6EF,'H3'],[0xB6F0,0xB6F0,'H2']])
call extend(s:tmp, [[0xB6F1,0xB70B,'H3'],[0xB70C,0xB70C,'H2'],[0xB70D,0xB727,'H3']])
call extend(s:tmp, [[0xB728,0xB728,'H2'],[0xB729,0xB743,'H3'],[0xB744,0xB744,'H2']])
call extend(s:tmp, [[0xB745,0xB75F,'H3'],[0xB760,0xB760,'H2'],[0xB761,0xB77B,'H3']])
call extend(s:tmp, [[0xB77C,0xB77C,'H2'],[0xB77D,0xB797,'H3'],[0xB798,0xB798,'H2']])
call extend(s:tmp, [[0xB799,0xB7B3,'H3'],[0xB7B4,0xB7B4,'H2'],[0xB7B5,0xB7CF,'H3']])
call extend(s:tmp, [[0xB7D0,0xB7D0,'H2'],[0xB7D1,0xB7EB,'H3'],[0xB7EC,0xB7EC,'H2']])
call extend(s:tmp, [[0xB7ED,0xB807,'H3'],[0xB808,0xB808,'H2'],[0xB809,0xB823,'H3']])
call extend(s:tmp, [[0xB824,0xB824,'H2'],[0xB825,0xB83F,'H3'],[0xB840,0xB840,'H2']])
call extend(s:tmp, [[0xB841,0xB85B,'H3'],[0xB85C,0xB85C,'H2'],[0xB85D,0xB877,'H3']])
call extend(s:tmp, [[0xB878,0xB878,'H2'],[0xB879,0xB893,'H3'],[0xB894,0xB894,'H2']])
call extend(s:tmp, [[0xB895,0xB8AF,'H3'],[0xB8B0,0xB8B0,'H2'],[0xB8B1,0xB8CB,'H3']])
call extend(s:tmp, [[0xB8CC,0xB8CC,'H2'],[0xB8CD,0xB8E7,'H3'],[0xB8E8,0xB8E8,'H2']])
call extend(s:tmp, [[0xB8E9,0xB903,'H3'],[0xB904,0xB904,'H2'],[0xB905,0xB91F,'H3']])
call extend(s:tmp, [[0xB920,0xB920,'H2'],[0xB921,0xB93B,'H3'],[0xB93C,0xB93C,'H2']])
call extend(s:tmp, [[0xB93D,0xB957,'H3'],[0xB958,0xB958,'H2'],[0xB959,0xB973,'H3']])
call extend(s:tmp, [[0xB974,0xB974,'H2'],[0xB975,0xB98F,'H3'],[0xB990,0xB990,'H2']])
call extend(s:tmp, [[0xB991,0xB9AB,'H3'],[0xB9AC,0xB9AC,'H2'],[0xB9AD,0xB9C7,'H3']])
call extend(s:tmp, [[0xB9C8,0xB9C8,'H2'],[0xB9C9,0xB9E3,'H3'],[0xB9E4,0xB9E4,'H2']])
call extend(s:tmp, [[0xB9E5,0xB9FF,'H3'],[0xBA00,0xBA00,'H2'],[0xBA01,0xBA1B,'H3']])
call extend(s:tmp, [[0xBA1C,0xBA1C,'H2'],[0xBA1D,0xBA37,'H3'],[0xBA38,0xBA38,'H2']])
call extend(s:tmp, [[0xBA39,0xBA53,'H3'],[0xBA54,0xBA54,'H2'],[0xBA55,0xBA6F,'H3']])
call extend(s:tmp, [[0xBA70,0xBA70,'H2'],[0xBA71,0xBA8B,'H3'],[0xBA8C,0xBA8C,'H2']])
call extend(s:tmp, [[0xBA8D,0xBAA7,'H3'],[0xBAA8,0xBAA8,'H2'],[0xBAA9,0xBAC3,'H3']])
call extend(s:tmp, [[0xBAC4,0xBAC4,'H2'],[0xBAC5,0xBADF,'H3'],[0xBAE0,0xBAE0,'H2']])
call extend(s:tmp, [[0xBAE1,0xBAFB,'H3'],[0xBAFC,0xBAFC,'H2'],[0xBAFD,0xBB17,'H3']])
call extend(s:tmp, [[0xBB18,0xBB18,'H2'],[0xBB19,0xBB33,'H3'],[0xBB34,0xBB34,'H2']])
call extend(s:tmp, [[0xBB35,0xBB4F,'H3'],[0xBB50,0xBB50,'H2'],[0xBB51,0xBB6B,'H3']])
call extend(s:tmp, [[0xBB6C,0xBB6C,'H2'],[0xBB6D,0xBB87,'H3'],[0xBB88,0xBB88,'H2']])
call extend(s:tmp, [[0xBB89,0xBBA3,'H3'],[0xBBA4,0xBBA4,'H2'],[0xBBA5,0xBBBF,'H3']])
call extend(s:tmp, [[0xBBC0,0xBBC0,'H2'],[0xBBC1,0xBBDB,'H3'],[0xBBDC,0xBBDC,'H2']])
call extend(s:tmp, [[0xBBDD,0xBBF7,'H3'],[0xBBF8,0xBBF8,'H2'],[0xBBF9,0xBC13,'H3']])
call extend(s:tmp, [[0xBC14,0xBC14,'H2'],[0xBC15,0xBC2F,'H3'],[0xBC30,0xBC30,'H2']])
call extend(s:tmp, [[0xBC31,0xBC4B,'H3'],[0xBC4C,0xBC4C,'H2'],[0xBC4D,0xBC67,'H3']])
call extend(s:tmp, [[0xBC68,0xBC68,'H2'],[0xBC69,0xBC83,'H3'],[0xBC84,0xBC84,'H2']])
call extend(s:tmp, [[0xBC85,0xBC9F,'H3'],[0xBCA0,0xBCA0,'H2'],[0xBCA1,0xBCBB,'H3']])
call extend(s:tmp, [[0xBCBC,0xBCBC,'H2'],[0xBCBD,0xBCD7,'H3'],[0xBCD8,0xBCD8,'H2']])
call extend(s:tmp, [[0xBCD9,0xBCF3,'H3'],[0xBCF4,0xBCF4,'H2'],[0xBCF5,0xBD0F,'H3']])
call extend(s:tmp, [[0xBD10,0xBD10,'H2'],[0xBD11,0xBD2B,'H3'],[0xBD2C,0xBD2C,'H2']])
call extend(s:tmp, [[0xBD2D,0xBD47,'H3'],[0xBD48,0xBD48,'H2'],[0xBD49,0xBD63,'H3']])
call extend(s:tmp, [[0xBD64,0xBD64,'H2'],[0xBD65,0xBD7F,'H3'],[0xBD80,0xBD80,'H2']])
call extend(s:tmp, [[0xBD81,0xBD9B,'H3'],[0xBD9C,0xBD9C,'H2'],[0xBD9D,0xBDB7,'H3']])
call extend(s:tmp, [[0xBDB8,0xBDB8,'H2'],[0xBDB9,0xBDD3,'H3'],[0xBDD4,0xBDD4,'H2']])
call extend(s:tmp, [[0xBDD5,0xBDEF,'H3'],[0xBDF0,0xBDF0,'H2'],[0xBDF1,0xBE0B,'H3']])
call extend(s:tmp, [[0xBE0C,0xBE0C,'H2'],[0xBE0D,0xBE27,'H3'],[0xBE28,0xBE28,'H2']])
call extend(s:tmp, [[0xBE29,0xBE43,'H3'],[0xBE44,0xBE44,'H2'],[0xBE45,0xBE5F,'H3']])
call extend(s:tmp, [[0xBE60,0xBE60,'H2'],[0xBE61,0xBE7B,'H3'],[0xBE7C,0xBE7C,'H2']])
call extend(s:tmp, [[0xBE7D,0xBE97,'H3'],[0xBE98,0xBE98,'H2'],[0xBE99,0xBEB3,'H3']])
call extend(s:tmp, [[0xBEB4,0xBEB4,'H2'],[0xBEB5,0xBECF,'H3'],[0xBED0,0xBED0,'H2']])
call extend(s:tmp, [[0xBED1,0xBEEB,'H3'],[0xBEEC,0xBEEC,'H2'],[0xBEED,0xBF07,'H3']])
call extend(s:tmp, [[0xBF08,0xBF08,'H2'],[0xBF09,0xBF23,'H3'],[0xBF24,0xBF24,'H2']])
call extend(s:tmp, [[0xBF25,0xBF3F,'H3'],[0xBF40,0xBF40,'H2'],[0xBF41,0xBF5B,'H3']])
call extend(s:tmp, [[0xBF5C,0xBF5C,'H2'],[0xBF5D,0xBF77,'H3'],[0xBF78,0xBF78,'H2']])
call extend(s:tmp, [[0xBF79,0xBF93,'H3'],[0xBF94,0xBF94,'H2'],[0xBF95,0xBFAF,'H3']])
call extend(s:tmp, [[0xBFB0,0xBFB0,'H2'],[0xBFB1,0xBFCB,'H3'],[0xBFCC,0xBFCC,'H2']])
call extend(s:tmp, [[0xBFCD,0xBFE7,'H3'],[0xBFE8,0xBFE8,'H2'],[0xBFE9,0xC003,'H3']])
call extend(s:tmp, [[0xC004,0xC004,'H2'],[0xC005,0xC01F,'H3'],[0xC020,0xC020,'H2']])
call extend(s:tmp, [[0xC021,0xC03B,'H3'],[0xC03C,0xC03C,'H2'],[0xC03D,0xC057,'H3']])
call extend(s:tmp, [[0xC058,0xC058,'H2'],[0xC059,0xC073,'H3'],[0xC074,0xC074,'H2']])
call extend(s:tmp, [[0xC075,0xC08F,'H3'],[0xC090,0xC090,'H2'],[0xC091,0xC0AB,'H3']])
call extend(s:tmp, [[0xC0AC,0xC0AC,'H2'],[0xC0AD,0xC0C7,'H3'],[0xC0C8,0xC0C8,'H2']])
call extend(s:tmp, [[0xC0C9,0xC0E3,'H3'],[0xC0E4,0xC0E4,'H2'],[0xC0E5,0xC0FF,'H3']])
call extend(s:tmp, [[0xC100,0xC100,'H2'],[0xC101,0xC11B,'H3'],[0xC11C,0xC11C,'H2']])
call extend(s:tmp, [[0xC11D,0xC137,'H3'],[0xC138,0xC138,'H2'],[0xC139,0xC153,'H3']])
call extend(s:tmp, [[0xC154,0xC154,'H2'],[0xC155,0xC16F,'H3'],[0xC170,0xC170,'H2']])
call extend(s:tmp, [[0xC171,0xC18B,'H3'],[0xC18C,0xC18C,'H2'],[0xC18D,0xC1A7,'H3']])
call extend(s:tmp, [[0xC1A8,0xC1A8,'H2'],[0xC1A9,0xC1C3,'H3'],[0xC1C4,0xC1C4,'H2']])
call extend(s:tmp, [[0xC1C5,0xC1DF,'H3'],[0xC1E0,0xC1E0,'H2'],[0xC1E1,0xC1FB,'H3']])
call extend(s:tmp, [[0xC1FC,0xC1FC,'H2'],[0xC1FD,0xC217,'H3'],[0xC218,0xC218,'H2']])
call extend(s:tmp, [[0xC219,0xC233,'H3'],[0xC234,0xC234,'H2'],[0xC235,0xC24F,'H3']])
call extend(s:tmp, [[0xC250,0xC250,'H2'],[0xC251,0xC26B,'H3'],[0xC26C,0xC26C,'H2']])
call extend(s:tmp, [[0xC26D,0xC287,'H3'],[0xC288,0xC288,'H2'],[0xC289,0xC2A3,'H3']])
call extend(s:tmp, [[0xC2A4,0xC2A4,'H2'],[0xC2A5,0xC2BF,'H3'],[0xC2C0,0xC2C0,'H2']])
call extend(s:tmp, [[0xC2C1,0xC2DB,'H3'],[0xC2DC,0xC2DC,'H2'],[0xC2DD,0xC2F7,'H3']])
call extend(s:tmp, [[0xC2F8,0xC2F8,'H2'],[0xC2F9,0xC313,'H3'],[0xC314,0xC314,'H2']])
call extend(s:tmp, [[0xC315,0xC32F,'H3'],[0xC330,0xC330,'H2'],[0xC331,0xC34B,'H3']])
call extend(s:tmp, [[0xC34C,0xC34C,'H2'],[0xC34D,0xC367,'H3'],[0xC368,0xC368,'H2']])
call extend(s:tmp, [[0xC369,0xC383,'H3'],[0xC384,0xC384,'H2'],[0xC385,0xC39F,'H3']])
call extend(s:tmp, [[0xC3A0,0xC3A0,'H2'],[0xC3A1,0xC3BB,'H3'],[0xC3BC,0xC3BC,'H2']])
call extend(s:tmp, [[0xC3BD,0xC3D7,'H3'],[0xC3D8,0xC3D8,'H2'],[0xC3D9,0xC3F3,'H3']])
call extend(s:tmp, [[0xC3F4,0xC3F4,'H2'],[0xC3F5,0xC40F,'H3'],[0xC410,0xC410,'H2']])
call extend(s:tmp, [[0xC411,0xC42B,'H3'],[0xC42C,0xC42C,'H2'],[0xC42D,0xC447,'H3']])
call extend(s:tmp, [[0xC448,0xC448,'H2'],[0xC449,0xC463,'H3'],[0xC464,0xC464,'H2']])
call extend(s:tmp, [[0xC465,0xC47F,'H3'],[0xC480,0xC480,'H2'],[0xC481,0xC49B,'H3']])
call extend(s:tmp, [[0xC49C,0xC49C,'H2'],[0xC49D,0xC4B7,'H3'],[0xC4B8,0xC4B8,'H2']])
call extend(s:tmp, [[0xC4B9,0xC4D3,'H3'],[0xC4D4,0xC4D4,'H2'],[0xC4D5,0xC4EF,'H3']])
call extend(s:tmp, [[0xC4F0,0xC4F0,'H2'],[0xC4F1,0xC50B,'H3'],[0xC50C,0xC50C,'H2']])
call extend(s:tmp, [[0xC50D,0xC527,'H3'],[0xC528,0xC528,'H2'],[0xC529,0xC543,'H3']])
call extend(s:tmp, [[0xC544,0xC544,'H2'],[0xC545,0xC55F,'H3'],[0xC560,0xC560,'H2']])
call extend(s:tmp, [[0xC561,0xC57B,'H3'],[0xC57C,0xC57C,'H2'],[0xC57D,0xC597,'H3']])
call extend(s:tmp, [[0xC598,0xC598,'H2'],[0xC599,0xC5B3,'H3'],[0xC5B4,0xC5B4,'H2']])
call extend(s:tmp, [[0xC5B5,0xC5CF,'H3'],[0xC5D0,0xC5D0,'H2'],[0xC5D1,0xC5EB,'H3']])
call extend(s:tmp, [[0xC5EC,0xC5EC,'H2'],[0xC5ED,0xC607,'H3'],[0xC608,0xC608,'H2']])
call extend(s:tmp, [[0xC609,0xC623,'H3'],[0xC624,0xC624,'H2'],[0xC625,0xC63F,'H3']])
call extend(s:tmp, [[0xC640,0xC640,'H2'],[0xC641,0xC65B,'H3'],[0xC65C,0xC65C,'H2']])
call extend(s:tmp, [[0xC65D,0xC677,'H3'],[0xC678,0xC678,'H2'],[0xC679,0xC693,'H3']])
call extend(s:tmp, [[0xC694,0xC694,'H2'],[0xC695,0xC6AF,'H3'],[0xC6B0,0xC6B0,'H2']])
call extend(s:tmp, [[0xC6B1,0xC6CB,'H3'],[0xC6CC,0xC6CC,'H2'],[0xC6CD,0xC6E7,'H3']])
call extend(s:tmp, [[0xC6E8,0xC6E8,'H2'],[0xC6E9,0xC703,'H3'],[0xC704,0xC704,'H2']])
call extend(s:tmp, [[0xC705,0xC71F,'H3'],[0xC720,0xC720,'H2'],[0xC721,0xC73B,'H3']])
call extend(s:tmp, [[0xC73C,0xC73C,'H2'],[0xC73D,0xC757,'H3'],[0xC758,0xC758,'H2']])
call extend(s:tmp, [[0xC759,0xC773,'H3'],[0xC774,0xC774,'H2'],[0xC775,0xC78F,'H3']])
call extend(s:tmp, [[0xC790,0xC790,'H2'],[0xC791,0xC7AB,'H3'],[0xC7AC,0xC7AC,'H2']])
call extend(s:tmp, [[0xC7AD,0xC7C7,'H3'],[0xC7C8,0xC7C8,'H2'],[0xC7C9,0xC7E3,'H3']])
call extend(s:tmp, [[0xC7E4,0xC7E4,'H2'],[0xC7E5,0xC7FF,'H3'],[0xC800,0xC800,'H2']])
call extend(s:tmp, [[0xC801,0xC81B,'H3'],[0xC81C,0xC81C,'H2'],[0xC81D,0xC837,'H3']])
call extend(s:tmp, [[0xC838,0xC838,'H2'],[0xC839,0xC853,'H3'],[0xC854,0xC854,'H2']])
call extend(s:tmp, [[0xC855,0xC86F,'H3'],[0xC870,0xC870,'H2'],[0xC871,0xC88B,'H3']])
call extend(s:tmp, [[0xC88C,0xC88C,'H2'],[0xC88D,0xC8A7,'H3'],[0xC8A8,0xC8A8,'H2']])
call extend(s:tmp, [[0xC8A9,0xC8C3,'H3'],[0xC8C4,0xC8C4,'H2'],[0xC8C5,0xC8DF,'H3']])
call extend(s:tmp, [[0xC8E0,0xC8E0,'H2'],[0xC8E1,0xC8FB,'H3'],[0xC8FC,0xC8FC,'H2']])
call extend(s:tmp, [[0xC8FD,0xC917,'H3'],[0xC918,0xC918,'H2'],[0xC919,0xC933,'H3']])
call extend(s:tmp, [[0xC934,0xC934,'H2'],[0xC935,0xC94F,'H3'],[0xC950,0xC950,'H2']])
call extend(s:tmp, [[0xC951,0xC96B,'H3'],[0xC96C,0xC96C,'H2'],[0xC96D,0xC987,'H3']])
call extend(s:tmp, [[0xC988,0xC988,'H2'],[0xC989,0xC9A3,'H3'],[0xC9A4,0xC9A4,'H2']])
call extend(s:tmp, [[0xC9A5,0xC9BF,'H3'],[0xC9C0,0xC9C0,'H2'],[0xC9C1,0xC9DB,'H3']])
call extend(s:tmp, [[0xC9DC,0xC9DC,'H2'],[0xC9DD,0xC9F7,'H3'],[0xC9F8,0xC9F8,'H2']])
call extend(s:tmp, [[0xC9F9,0xCA13,'H3'],[0xCA14,0xCA14,'H2'],[0xCA15,0xCA2F,'H3']])
call extend(s:tmp, [[0xCA30,0xCA30,'H2'],[0xCA31,0xCA4B,'H3'],[0xCA4C,0xCA4C,'H2']])
call extend(s:tmp, [[0xCA4D,0xCA67,'H3'],[0xCA68,0xCA68,'H2'],[0xCA69,0xCA83,'H3']])
call extend(s:tmp, [[0xCA84,0xCA84,'H2'],[0xCA85,0xCA9F,'H3'],[0xCAA0,0xCAA0,'H2']])
call extend(s:tmp, [[0xCAA1,0xCABB,'H3'],[0xCABC,0xCABC,'H2'],[0xCABD,0xCAD7,'H3']])
call extend(s:tmp, [[0xCAD8,0xCAD8,'H2'],[0xCAD9,0xCAF3,'H3'],[0xCAF4,0xCAF4,'H2']])
call extend(s:tmp, [[0xCAF5,0xCB0F,'H3'],[0xCB10,0xCB10,'H2'],[0xCB11,0xCB2B,'H3']])
call extend(s:tmp, [[0xCB2C,0xCB2C,'H2'],[0xCB2D,0xCB47,'H3'],[0xCB48,0xCB48,'H2']])
call extend(s:tmp, [[0xCB49,0xCB63,'H3'],[0xCB64,0xCB64,'H2'],[0xCB65,0xCB7F,'H3']])
call extend(s:tmp, [[0xCB80,0xCB80,'H2'],[0xCB81,0xCB9B,'H3'],[0xCB9C,0xCB9C,'H2']])
call extend(s:tmp, [[0xCB9D,0xCBB7,'H3'],[0xCBB8,0xCBB8,'H2'],[0xCBB9,0xCBD3,'H3']])
call extend(s:tmp, [[0xCBD4,0xCBD4,'H2'],[0xCBD5,0xCBEF,'H3'],[0xCBF0,0xCBF0,'H2']])
call extend(s:tmp, [[0xCBF1,0xCC0B,'H3'],[0xCC0C,0xCC0C,'H2'],[0xCC0D,0xCC27,'H3']])
call extend(s:tmp, [[0xCC28,0xCC28,'H2'],[0xCC29,0xCC43,'H3'],[0xCC44,0xCC44,'H2']])
call extend(s:tmp, [[0xCC45,0xCC5F,'H3'],[0xCC60,0xCC60,'H2'],[0xCC61,0xCC7B,'H3']])
call extend(s:tmp, [[0xCC7C,0xCC7C,'H2'],[0xCC7D,0xCC97,'H3'],[0xCC98,0xCC98,'H2']])
call extend(s:tmp, [[0xCC99,0xCCB3,'H3'],[0xCCB4,0xCCB4,'H2'],[0xCCB5,0xCCCF,'H3']])
call extend(s:tmp, [[0xCCD0,0xCCD0,'H2'],[0xCCD1,0xCCEB,'H3'],[0xCCEC,0xCCEC,'H2']])
call extend(s:tmp, [[0xCCED,0xCD07,'H3'],[0xCD08,0xCD08,'H2'],[0xCD09,0xCD23,'H3']])
call extend(s:tmp, [[0xCD24,0xCD24,'H2'],[0xCD25,0xCD3F,'H3'],[0xCD40,0xCD40,'H2']])
call extend(s:tmp, [[0xCD41,0xCD5B,'H3'],[0xCD5C,0xCD5C,'H2'],[0xCD5D,0xCD77,'H3']])
call extend(s:tmp, [[0xCD78,0xCD78,'H2'],[0xCD79,0xCD93,'H3'],[0xCD94,0xCD94,'H2']])
call extend(s:tmp, [[0xCD95,0xCDAF,'H3'],[0xCDB0,0xCDB0,'H2'],[0xCDB1,0xCDCB,'H3']])
call extend(s:tmp, [[0xCDCC,0xCDCC,'H2'],[0xCDCD,0xCDE7,'H3'],[0xCDE8,0xCDE8,'H2']])
call extend(s:tmp, [[0xCDE9,0xCE03,'H3'],[0xCE04,0xCE04,'H2'],[0xCE05,0xCE1F,'H3']])
call extend(s:tmp, [[0xCE20,0xCE20,'H2'],[0xCE21,0xCE3B,'H3'],[0xCE3C,0xCE3C,'H2']])
call extend(s:tmp, [[0xCE3D,0xCE57,'H3'],[0xCE58,0xCE58,'H2'],[0xCE59,0xCE73,'H3']])
call extend(s:tmp, [[0xCE74,0xCE74,'H2'],[0xCE75,0xCE8F,'H3'],[0xCE90,0xCE90,'H2']])
call extend(s:tmp, [[0xCE91,0xCEAB,'H3'],[0xCEAC,0xCEAC,'H2'],[0xCEAD,0xCEC7,'H3']])
call extend(s:tmp, [[0xCEC8,0xCEC8,'H2'],[0xCEC9,0xCEE3,'H3'],[0xCEE4,0xCEE4,'H2']])
call extend(s:tmp, [[0xCEE5,0xCEFF,'H3'],[0xCF00,0xCF00,'H2'],[0xCF01,0xCF1B,'H3']])
call extend(s:tmp, [[0xCF1C,0xCF1C,'H2'],[0xCF1D,0xCF37,'H3'],[0xCF38,0xCF38,'H2']])
call extend(s:tmp, [[0xCF39,0xCF53,'H3'],[0xCF54,0xCF54,'H2'],[0xCF55,0xCF6F,'H3']])
call extend(s:tmp, [[0xCF70,0xCF70,'H2'],[0xCF71,0xCF8B,'H3'],[0xCF8C,0xCF8C,'H2']])
call extend(s:tmp, [[0xCF8D,0xCFA7,'H3'],[0xCFA8,0xCFA8,'H2'],[0xCFA9,0xCFC3,'H3']])
call extend(s:tmp, [[0xCFC4,0xCFC4,'H2'],[0xCFC5,0xCFDF,'H3'],[0xCFE0,0xCFE0,'H2']])
call extend(s:tmp, [[0xCFE1,0xCFFB,'H3'],[0xCFFC,0xCFFC,'H2'],[0xCFFD,0xD017,'H3']])
call extend(s:tmp, [[0xD018,0xD018,'H2'],[0xD019,0xD033,'H3'],[0xD034,0xD034,'H2']])
call extend(s:tmp, [[0xD035,0xD04F,'H3'],[0xD050,0xD050,'H2'],[0xD051,0xD06B,'H3']])
call extend(s:tmp, [[0xD06C,0xD06C,'H2'],[0xD06D,0xD087,'H3'],[0xD088,0xD088,'H2']])
call extend(s:tmp, [[0xD089,0xD0A3,'H3'],[0xD0A4,0xD0A4,'H2'],[0xD0A5,0xD0BF,'H3']])
call extend(s:tmp, [[0xD0C0,0xD0C0,'H2'],[0xD0C1,0xD0DB,'H3'],[0xD0DC,0xD0DC,'H2']])
call extend(s:tmp, [[0xD0DD,0xD0F7,'H3'],[0xD0F8,0xD0F8,'H2'],[0xD0F9,0xD113,'H3']])
call extend(s:tmp, [[0xD114,0xD114,'H2'],[0xD115,0xD12F,'H3'],[0xD130,0xD130,'H2']])
call extend(s:tmp, [[0xD131,0xD14B,'H3'],[0xD14C,0xD14C,'H2'],[0xD14D,0xD167,'H3']])
call extend(s:tmp, [[0xD168,0xD168,'H2'],[0xD169,0xD183,'H3'],[0xD184,0xD184,'H2']])
call extend(s:tmp, [[0xD185,0xD19F,'H3'],[0xD1A0,0xD1A0,'H2'],[0xD1A1,0xD1BB,'H3']])
call extend(s:tmp, [[0xD1BC,0xD1BC,'H2'],[0xD1BD,0xD1D7,'H3'],[0xD1D8,0xD1D8,'H2']])
call extend(s:tmp, [[0xD1D9,0xD1F3,'H3'],[0xD1F4,0xD1F4,'H2'],[0xD1F5,0xD20F,'H3']])
call extend(s:tmp, [[0xD210,0xD210,'H2'],[0xD211,0xD22B,'H3'],[0xD22C,0xD22C,'H2']])
call extend(s:tmp, [[0xD22D,0xD247,'H3'],[0xD248,0xD248,'H2'],[0xD249,0xD263,'H3']])
call extend(s:tmp, [[0xD264,0xD264,'H2'],[0xD265,0xD27F,'H3'],[0xD280,0xD280,'H2']])
call extend(s:tmp, [[0xD281,0xD29B,'H3'],[0xD29C,0xD29C,'H2'],[0xD29D,0xD2B7,'H3']])
call extend(s:tmp, [[0xD2B8,0xD2B8,'H2'],[0xD2B9,0xD2D3,'H3'],[0xD2D4,0xD2D4,'H2']])
call extend(s:tmp, [[0xD2D5,0xD2EF,'H3'],[0xD2F0,0xD2F0,'H2'],[0xD2F1,0xD30B,'H3']])
call extend(s:tmp, [[0xD30C,0xD30C,'H2'],[0xD30D,0xD327,'H3'],[0xD328,0xD328,'H2']])
call extend(s:tmp, [[0xD329,0xD343,'H3'],[0xD344,0xD344,'H2'],[0xD345,0xD35F,'H3']])
call extend(s:tmp, [[0xD360,0xD360,'H2'],[0xD361,0xD37B,'H3'],[0xD37C,0xD37C,'H2']])
call extend(s:tmp, [[0xD37D,0xD397,'H3'],[0xD398,0xD398,'H2'],[0xD399,0xD3B3,'H3']])
call extend(s:tmp, [[0xD3B4,0xD3B4,'H2'],[0xD3B5,0xD3CF,'H3'],[0xD3D0,0xD3D0,'H2']])
call extend(s:tmp, [[0xD3D1,0xD3EB,'H3'],[0xD3EC,0xD3EC,'H2'],[0xD3ED,0xD407,'H3']])
call extend(s:tmp, [[0xD408,0xD408,'H2'],[0xD409,0xD423,'H3'],[0xD424,0xD424,'H2']])
call extend(s:tmp, [[0xD425,0xD43F,'H3'],[0xD440,0xD440,'H2'],[0xD441,0xD45B,'H3']])
call extend(s:tmp, [[0xD45C,0xD45C,'H2'],[0xD45D,0xD477,'H3'],[0xD478,0xD478,'H2']])
call extend(s:tmp, [[0xD479,0xD493,'H3'],[0xD494,0xD494,'H2'],[0xD495,0xD4AF,'H3']])
call extend(s:tmp, [[0xD4B0,0xD4B0,'H2'],[0xD4B1,0xD4CB,'H3'],[0xD4CC,0xD4CC,'H2']])
call extend(s:tmp, [[0xD4CD,0xD4E7,'H3'],[0xD4E8,0xD4E8,'H2'],[0xD4E9,0xD503,'H3']])
call extend(s:tmp, [[0xD504,0xD504,'H2'],[0xD505,0xD51F,'H3'],[0xD520,0xD520,'H2']])
call extend(s:tmp, [[0xD521,0xD53B,'H3'],[0xD53C,0xD53C,'H2'],[0xD53D,0xD557,'H3']])
call extend(s:tmp, [[0xD558,0xD558,'H2'],[0xD559,0xD573,'H3'],[0xD574,0xD574,'H2']])
call extend(s:tmp, [[0xD575,0xD58F,'H3'],[0xD590,0xD590,'H2'],[0xD591,0xD5AB,'H3']])
call extend(s:tmp, [[0xD5AC,0xD5AC,'H2'],[0xD5AD,0xD5C7,'H3'],[0xD5C8,0xD5C8,'H2']])
call extend(s:tmp, [[0xD5C9,0xD5E3,'H3'],[0xD5E4,0xD5E4,'H2'],[0xD5E5,0xD5FF,'H3']])
call extend(s:tmp, [[0xD600,0xD600,'H2'],[0xD601,0xD61B,'H3'],[0xD61C,0xD61C,'H2']])
call extend(s:tmp, [[0xD61D,0xD637,'H3'],[0xD638,0xD638,'H2'],[0xD639,0xD653,'H3']])
call extend(s:tmp, [[0xD654,0xD654,'H2'],[0xD655,0xD66F,'H3'],[0xD670,0xD670,'H2']])
call extend(s:tmp, [[0xD671,0xD68B,'H3'],[0xD68C,0xD68C,'H2'],[0xD68D,0xD6A7,'H3']])
call extend(s:tmp, [[0xD6A8,0xD6A8,'H2'],[0xD6A9,0xD6C3,'H3'],[0xD6C4,0xD6C4,'H2']])
call extend(s:tmp, [[0xD6C5,0xD6DF,'H3'],[0xD6E0,0xD6E0,'H2'],[0xD6E1,0xD6FB,'H3']])
call extend(s:tmp, [[0xD6FC,0xD6FC,'H2'],[0xD6FD,0xD717,'H3'],[0xD718,0xD718,'H2']])
call extend(s:tmp, [[0xD719,0xD733,'H3'],[0xD734,0xD734,'H2'],[0xD735,0xD74F,'H3']])
call extend(s:tmp, [[0xD750,0xD750,'H2'],[0xD751,0xD76B,'H3'],[0xD76C,0xD76C,'H2']])
call extend(s:tmp, [[0xD76D,0xD787,'H3'],[0xD788,0xD788,'H2'],[0xD789,0xD7A3,'H3']])
call extend(s:tmp, [[0xD7B0,0xD7C6,'JV'],[0xD7CB,0xD7FB,'JT'],[0xD800,0xDFFF,'SG']])
call extend(s:tmp, [[0xE000,0xF8FF,'XX'],[0xF900,0xFAFF,'ID'],[0xFB00,0xFB06,'AL']])
call extend(s:tmp, [[0xFB13,0xFB17,'AL'],[0xFB1D,0xFB1D,'HL'],[0xFB1E,0xFB1E,'CM']])
call extend(s:tmp, [[0xFB1F,0xFB28,'HL'],[0xFB29,0xFB29,'AL'],[0xFB2A,0xFB36,'HL']])
call extend(s:tmp, [[0xFB38,0xFB3C,'HL'],[0xFB3E,0xFB3E,'HL'],[0xFB40,0xFB41,'HL']])
call extend(s:tmp, [[0xFB43,0xFB44,'HL'],[0xFB46,0xFB4F,'HL'],[0xFB50,0xFBC1,'AL']])
call extend(s:tmp, [[0xFBD3,0xFD3D,'AL'],[0xFD3E,0xFD3E,'CL'],[0xFD3F,0xFD3F,'OP']])
call extend(s:tmp, [[0xFD50,0xFD8F,'AL'],[0xFD92,0xFDC7,'AL'],[0xFDF0,0xFDFB,'AL']])
call extend(s:tmp, [[0xFDFC,0xFDFC,'PO'],[0xFDFD,0xFDFD,'AL'],[0xFE00,0xFE0F,'CM']])
call extend(s:tmp, [[0xFE10,0xFE10,'IS'],[0xFE11,0xFE12,'CL'],[0xFE13,0xFE14,'IS']])
call extend(s:tmp, [[0xFE15,0xFE16,'EX'],[0xFE17,0xFE17,'OP'],[0xFE18,0xFE18,'CL']])
call extend(s:tmp, [[0xFE19,0xFE19,'IN'],[0xFE20,0xFE2F,'CM'],[0xFE30,0xFE34,'ID']])
call extend(s:tmp, [[0xFE35,0xFE35,'OP'],[0xFE36,0xFE36,'CL'],[0xFE37,0xFE37,'OP']])
call extend(s:tmp, [[0xFE38,0xFE38,'CL'],[0xFE39,0xFE39,'OP'],[0xFE3A,0xFE3A,'CL']])
call extend(s:tmp, [[0xFE3B,0xFE3B,'OP'],[0xFE3C,0xFE3C,'CL'],[0xFE3D,0xFE3D,'OP']])
call extend(s:tmp, [[0xFE3E,0xFE3E,'CL'],[0xFE3F,0xFE3F,'OP'],[0xFE40,0xFE40,'CL']])
call extend(s:tmp, [[0xFE41,0xFE41,'OP'],[0xFE42,0xFE42,'CL'],[0xFE43,0xFE43,'OP']])
call extend(s:tmp, [[0xFE44,0xFE44,'CL'],[0xFE45,0xFE46,'ID'],[0xFE47,0xFE47,'OP']])
call extend(s:tmp, [[0xFE48,0xFE48,'CL'],[0xFE49,0xFE4F,'ID'],[0xFE50,0xFE50,'CL']])
call extend(s:tmp, [[0xFE51,0xFE51,'ID'],[0xFE52,0xFE52,'CL'],[0xFE54,0xFE55,'NS']])
call extend(s:tmp, [[0xFE56,0xFE57,'EX'],[0xFE58,0xFE58,'ID'],[0xFE59,0xFE59,'OP']])
call extend(s:tmp, [[0xFE5A,0xFE5A,'CL'],[0xFE5B,0xFE5B,'OP'],[0xFE5C,0xFE5C,'CL']])
call extend(s:tmp, [[0xFE5D,0xFE5D,'OP'],[0xFE5E,0xFE5E,'CL'],[0xFE5F,0xFE66,'ID']])
call extend(s:tmp, [[0xFE68,0xFE68,'ID'],[0xFE69,0xFE69,'PR'],[0xFE6A,0xFE6A,'PO']])
call extend(s:tmp, [[0xFE6B,0xFE6B,'ID'],[0xFE70,0xFE74,'AL'],[0xFE76,0xFEFC,'AL']])
call extend(s:tmp, [[0xFEFF,0xFEFF,'WJ'],[0xFF01,0xFF01,'EX'],[0xFF02,0xFF03,'ID']])
call extend(s:tmp, [[0xFF04,0xFF04,'PR'],[0xFF05,0xFF05,'PO'],[0xFF06,0xFF07,'ID']])
call extend(s:tmp, [[0xFF08,0xFF08,'OP'],[0xFF09,0xFF09,'CL'],[0xFF0A,0xFF0B,'ID']])
call extend(s:tmp, [[0xFF0C,0xFF0C,'CL'],[0xFF0D,0xFF0D,'ID'],[0xFF0E,0xFF0E,'CL']])
call extend(s:tmp, [[0xFF0F,0xFF19,'ID'],[0xFF1A,0xFF1B,'NS'],[0xFF1C,0xFF1E,'ID']])
call extend(s:tmp, [[0xFF1F,0xFF1F,'EX'],[0xFF20,0xFF3A,'ID'],[0xFF3B,0xFF3B,'OP']])
call extend(s:tmp, [[0xFF3C,0xFF3C,'ID'],[0xFF3D,0xFF3D,'CL'],[0xFF3E,0xFF5A,'ID']])
call extend(s:tmp, [[0xFF5B,0xFF5B,'OP'],[0xFF5C,0xFF5C,'ID'],[0xFF5D,0xFF5D,'CL']])
call extend(s:tmp, [[0xFF5E,0xFF5E,'ID'],[0xFF5F,0xFF5F,'OP'],[0xFF60,0xFF61,'CL']])
call extend(s:tmp, [[0xFF62,0xFF62,'OP'],[0xFF63,0xFF64,'CL'],[0xFF65,0xFF65,'NS']])
call extend(s:tmp, [[0xFF66,0xFF66,'ID'],[0xFF67,0xFF70,'CJ'],[0xFF71,0xFF9D,'ID']])
call extend(s:tmp, [[0xFF9E,0xFF9F,'NS'],[0xFFA0,0xFFBE,'ID'],[0xFFC2,0xFFC7,'ID']])
call extend(s:tmp, [[0xFFCA,0xFFCF,'ID'],[0xFFD2,0xFFD7,'ID'],[0xFFDA,0xFFDC,'ID']])
call extend(s:tmp, [[0xFFE0,0xFFE0,'PO'],[0xFFE1,0xFFE1,'PR'],[0xFFE2,0xFFE4,'ID']])
call extend(s:tmp, [[0xFFE5,0xFFE6,'PR'],[0xFFE8,0xFFEE,'AL'],[0xFFF9,0xFFFB,'CM']])
call extend(s:tmp, [[0xFFFC,0xFFFC,'CB'],[0xFFFD,0xFFFD,'AI'],[0x10000,0x1000B,'AL']])
call extend(s:tmp, [[0x1000D,0x10026,'AL'],[0x10028,0x1003A,'AL'],[0x1003C,0x1003D,'AL']])
call extend(s:tmp, [[0x1003F,0x1004D,'AL'],[0x10050,0x1005D,'AL'],[0x10080,0x100FA,'AL']])
call extend(s:tmp, [[0x10100,0x10102,'BA'],[0x10107,0x10133,'AL'],[0x10137,0x1018E,'AL']])
call extend(s:tmp, [[0x10190,0x1019B,'AL'],[0x101A0,0x101A0,'AL'],[0x101D0,0x101FC,'AL']])
call extend(s:tmp, [[0x101FD,0x101FD,'CM'],[0x10280,0x1029C,'AL'],[0x102A0,0x102D0,'AL']])
call extend(s:tmp, [[0x102E0,0x102E0,'CM'],[0x102E1,0x102FB,'AL'],[0x10300,0x10323,'AL']])
call extend(s:tmp, [[0x10330,0x1034A,'AL'],[0x10350,0x10375,'AL'],[0x10376,0x1037A,'CM']])
call extend(s:tmp, [[0x10380,0x1039D,'AL'],[0x1039F,0x1039F,'BA'],[0x103A0,0x103C3,'AL']])
call extend(s:tmp, [[0x103C8,0x103CF,'AL'],[0x103D0,0x103D0,'BA'],[0x103D1,0x103D5,'AL']])
call extend(s:tmp, [[0x10400,0x1049D,'AL'],[0x104A0,0x104A9,'NU'],[0x104B0,0x104D3,'AL']])
call extend(s:tmp, [[0x104D8,0x104FB,'AL'],[0x10500,0x10527,'AL'],[0x10530,0x10563,'AL']])
call extend(s:tmp, [[0x1056F,0x1056F,'AL'],[0x10600,0x10736,'AL'],[0x10740,0x10755,'AL']])
call extend(s:tmp, [[0x10760,0x10767,'AL'],[0x10800,0x10805,'AL'],[0x10808,0x10808,'AL']])
call extend(s:tmp, [[0x1080A,0x10835,'AL'],[0x10837,0x10838,'AL'],[0x1083C,0x1083C,'AL']])
call extend(s:tmp, [[0x1083F,0x10855,'AL'],[0x10857,0x10857,'BA'],[0x10858,0x1089E,'AL']])
call extend(s:tmp, [[0x108A7,0x108AF,'AL'],[0x108E0,0x108F2,'AL'],[0x108F4,0x108F5,'AL']])
call extend(s:tmp, [[0x108FB,0x1091B,'AL'],[0x1091F,0x1091F,'BA'],[0x10920,0x10939,'AL']])
call extend(s:tmp, [[0x1093F,0x1093F,'AL'],[0x10980,0x109B7,'AL'],[0x109BC,0x109CF,'AL']])
call extend(s:tmp, [[0x109D2,0x10A00,'AL'],[0x10A01,0x10A03,'CM'],[0x10A05,0x10A06,'CM']])
call extend(s:tmp, [[0x10A0C,0x10A0F,'CM'],[0x10A10,0x10A13,'AL'],[0x10A15,0x10A17,'AL']])
call extend(s:tmp, [[0x10A19,0x10A33,'AL'],[0x10A38,0x10A3A,'CM'],[0x10A3F,0x10A3F,'CM']])
call extend(s:tmp, [[0x10A40,0x10A47,'AL'],[0x10A50,0x10A57,'BA'],[0x10A58,0x10A58,'AL']])
call extend(s:tmp, [[0x10A60,0x10A9F,'AL'],[0x10AC0,0x10AE4,'AL'],[0x10AE5,0x10AE6,'CM']])
call extend(s:tmp, [[0x10AEB,0x10AEF,'AL'],[0x10AF0,0x10AF5,'BA'],[0x10AF6,0x10AF6,'IN']])
call extend(s:tmp, [[0x10B00,0x10B35,'AL'],[0x10B39,0x10B3F,'BA'],[0x10B40,0x10B55,'AL']])
call extend(s:tmp, [[0x10B58,0x10B72,'AL'],[0x10B78,0x10B91,'AL'],[0x10B99,0x10B9C,'AL']])
call extend(s:tmp, [[0x10BA9,0x10BAF,'AL'],[0x10C00,0x10C48,'AL'],[0x10C80,0x10CB2,'AL']])
call extend(s:tmp, [[0x10CC0,0x10CF2,'AL'],[0x10CFA,0x10CFF,'AL'],[0x10E60,0x10E7E,'AL']])
call extend(s:tmp, [[0x11000,0x11002,'CM'],[0x11003,0x11037,'AL'],[0x11038,0x11046,'CM']])
call extend(s:tmp, [[0x11047,0x11048,'BA'],[0x11049,0x1104D,'AL'],[0x11052,0x11065,'AL']])
call extend(s:tmp, [[0x11066,0x1106F,'NU'],[0x1107F,0x11082,'CM'],[0x11083,0x110AF,'AL']])
call extend(s:tmp, [[0x110B0,0x110BA,'CM'],[0x110BB,0x110BD,'AL'],[0x110BE,0x110C1,'BA']])
call extend(s:tmp, [[0x110D0,0x110E8,'AL'],[0x110F0,0x110F9,'NU'],[0x11100,0x11102,'CM']])
call extend(s:tmp, [[0x11103,0x11126,'AL'],[0x11127,0x11134,'CM'],[0x11136,0x1113F,'NU']])
call extend(s:tmp, [[0x11140,0x11143,'BA'],[0x11150,0x11172,'AL'],[0x11173,0x11173,'CM']])
call extend(s:tmp, [[0x11174,0x11174,'AL'],[0x11175,0x11175,'BB'],[0x11176,0x11176,'AL']])
call extend(s:tmp, [[0x11180,0x11182,'CM'],[0x11183,0x111B2,'AL'],[0x111B3,0x111C0,'CM']])
call extend(s:tmp, [[0x111C1,0x111C4,'AL'],[0x111C5,0x111C6,'BA'],[0x111C7,0x111C7,'AL']])
call extend(s:tmp, [[0x111C8,0x111C8,'BA'],[0x111C9,0x111C9,'AL'],[0x111CA,0x111CC,'CM']])
call extend(s:tmp, [[0x111CD,0x111CD,'AL'],[0x111D0,0x111D9,'NU'],[0x111DA,0x111DA,'AL']])
call extend(s:tmp, [[0x111DB,0x111DB,'BB'],[0x111DC,0x111DC,'AL'],[0x111DD,0x111DF,'BA']])
call extend(s:tmp, [[0x111E1,0x111F4,'AL'],[0x11200,0x11211,'AL'],[0x11213,0x1122B,'AL']])
call extend(s:tmp, [[0x1122C,0x11237,'CM'],[0x11238,0x11239,'BA'],[0x1123A,0x1123A,'AL']])
call extend(s:tmp, [[0x1123B,0x1123C,'BA'],[0x1123D,0x1123D,'AL'],[0x1123E,0x1123E,'CM']])
call extend(s:tmp, [[0x11280,0x11286,'AL'],[0x11288,0x11288,'AL'],[0x1128A,0x1128D,'AL']])
call extend(s:tmp, [[0x1128F,0x1129D,'AL'],[0x1129F,0x112A8,'AL'],[0x112A9,0x112A9,'BA']])
call extend(s:tmp, [[0x112B0,0x112DE,'AL'],[0x112DF,0x112EA,'CM'],[0x112F0,0x112F9,'NU']])
call extend(s:tmp, [[0x11300,0x11303,'CM'],[0x11305,0x1130C,'AL'],[0x1130F,0x11310,'AL']])
call extend(s:tmp, [[0x11313,0x11328,'AL'],[0x1132A,0x11330,'AL'],[0x11332,0x11333,'AL']])
call extend(s:tmp, [[0x11335,0x11339,'AL'],[0x1133C,0x1133C,'CM'],[0x1133D,0x1133D,'AL']])
call extend(s:tmp, [[0x1133E,0x11344,'CM'],[0x11347,0x11348,'CM'],[0x1134B,0x1134D,'CM']])
call extend(s:tmp, [[0x11350,0x11350,'AL'],[0x11357,0x11357,'CM'],[0x1135D,0x11361,'AL']])
call extend(s:tmp, [[0x11362,0x11363,'CM'],[0x11366,0x1136C,'CM'],[0x11370,0x11374,'CM']])
call extend(s:tmp, [[0x11400,0x11434,'AL'],[0x11435,0x11446,'CM'],[0x11447,0x1144A,'AL']])
call extend(s:tmp, [[0x1144B,0x1144E,'BA'],[0x1144F,0x1144F,'AL'],[0x11450,0x11459,'NU']])
call extend(s:tmp, [[0x1145B,0x1145B,'BA'],[0x1145D,0x1145D,'AL'],[0x11480,0x114AF,'AL']])
call extend(s:tmp, [[0x114B0,0x114C3,'CM'],[0x114C4,0x114C7,'AL'],[0x114D0,0x114D9,'NU']])
call extend(s:tmp, [[0x11580,0x115AE,'AL'],[0x115AF,0x115B5,'CM'],[0x115B8,0x115C0,'CM']])
call extend(s:tmp, [[0x115C1,0x115C1,'BB'],[0x115C2,0x115C3,'BA'],[0x115C4,0x115C5,'EX']])
call extend(s:tmp, [[0x115C6,0x115C8,'AL'],[0x115C9,0x115D7,'BA'],[0x115D8,0x115DB,'AL']])
call extend(s:tmp, [[0x115DC,0x115DD,'CM'],[0x11600,0x1162F,'AL'],[0x11630,0x11640,'CM']])
call extend(s:tmp, [[0x11641,0x11642,'BA'],[0x11643,0x11644,'AL'],[0x11650,0x11659,'NU']])
call extend(s:tmp, [[0x11660,0x1166C,'BB'],[0x11680,0x116AA,'AL'],[0x116AB,0x116B7,'CM']])
call extend(s:tmp, [[0x116C0,0x116C9,'NU'],[0x11700,0x11719,'SA'],[0x1171D,0x1172B,'SA']])
call extend(s:tmp, [[0x11730,0x11739,'NU'],[0x1173A,0x1173B,'SA'],[0x1173C,0x1173E,'BA']])
call extend(s:tmp, [[0x1173F,0x1173F,'SA'],[0x118A0,0x118DF,'AL'],[0x118E0,0x118E9,'NU']])
call extend(s:tmp, [[0x118EA,0x118F2,'AL'],[0x118FF,0x118FF,'AL'],[0x11AC0,0x11AF8,'AL']])
call extend(s:tmp, [[0x11C00,0x11C08,'AL'],[0x11C0A,0x11C2E,'AL'],[0x11C2F,0x11C36,'CM']])
call extend(s:tmp, [[0x11C38,0x11C3F,'CM'],[0x11C40,0x11C40,'AL'],[0x11C41,0x11C45,'BA']])
call extend(s:tmp, [[0x11C50,0x11C59,'NU'],[0x11C5A,0x11C6C,'AL'],[0x11C70,0x11C70,'BB']])
call extend(s:tmp, [[0x11C71,0x11C71,'EX'],[0x11C72,0x11C8F,'AL'],[0x11C92,0x11CA7,'CM']])
call extend(s:tmp, [[0x11CA9,0x11CB6,'CM'],[0x12000,0x12399,'AL'],[0x12400,0x1246E,'AL']])
call extend(s:tmp, [[0x12470,0x12474,'BA'],[0x12480,0x12543,'AL'],[0x13000,0x13257,'AL']])
call extend(s:tmp, [[0x13258,0x1325A,'OP'],[0x1325B,0x1325D,'CL'],[0x1325E,0x13281,'AL']])
call extend(s:tmp, [[0x13282,0x13282,'CL'],[0x13283,0x13285,'AL'],[0x13286,0x13286,'OP']])
call extend(s:tmp, [[0x13287,0x13287,'CL'],[0x13288,0x13288,'OP'],[0x13289,0x13289,'CL']])
call extend(s:tmp, [[0x1328A,0x13378,'AL'],[0x13379,0x13379,'OP'],[0x1337A,0x1337B,'CL']])
call extend(s:tmp, [[0x1337C,0x1342E,'AL'],[0x14400,0x145CD,'AL'],[0x145CE,0x145CE,'OP']])
call extend(s:tmp, [[0x145CF,0x145CF,'CL'],[0x145D0,0x14646,'AL'],[0x16800,0x16A38,'AL']])
call extend(s:tmp, [[0x16A40,0x16A5E,'AL'],[0x16A60,0x16A69,'NU'],[0x16A6E,0x16A6F,'BA']])
call extend(s:tmp, [[0x16AD0,0x16AED,'AL'],[0x16AF0,0x16AF4,'CM'],[0x16AF5,0x16AF5,'BA']])
call extend(s:tmp, [[0x16B00,0x16B2F,'AL'],[0x16B30,0x16B36,'CM'],[0x16B37,0x16B39,'BA']])
call extend(s:tmp, [[0x16B3A,0x16B43,'AL'],[0x16B44,0x16B44,'BA'],[0x16B45,0x16B45,'AL']])
call extend(s:tmp, [[0x16B50,0x16B59,'NU'],[0x16B5B,0x16B61,'AL'],[0x16B63,0x16B77,'AL']])
call extend(s:tmp, [[0x16B7D,0x16B8F,'AL'],[0x16F00,0x16F44,'AL'],[0x16F50,0x16F50,'AL']])
call extend(s:tmp, [[0x16F51,0x16F7E,'CM'],[0x16F8F,0x16F92,'CM'],[0x16F93,0x16F9F,'AL']])
call extend(s:tmp, [[0x16FE0,0x16FE0,'NS'],[0x17000,0x187EC,'ID'],[0x18800,0x18AF2,'ID']])
call extend(s:tmp, [[0x1B000,0x1B001,'ID'],[0x1BC00,0x1BC6A,'AL'],[0x1BC70,0x1BC7C,'AL']])
call extend(s:tmp, [[0x1BC80,0x1BC88,'AL'],[0x1BC90,0x1BC99,'AL'],[0x1BC9C,0x1BC9C,'AL']])
call extend(s:tmp, [[0x1BC9D,0x1BC9E,'CM'],[0x1BC9F,0x1BC9F,'BA'],[0x1BCA0,0x1BCA3,'CM']])
call extend(s:tmp, [[0x1D000,0x1D0F5,'AL'],[0x1D100,0x1D126,'AL'],[0x1D129,0x1D164,'AL']])
call extend(s:tmp, [[0x1D165,0x1D169,'CM'],[0x1D16A,0x1D16C,'AL'],[0x1D16D,0x1D182,'CM']])
call extend(s:tmp, [[0x1D183,0x1D184,'AL'],[0x1D185,0x1D18B,'CM'],[0x1D18C,0x1D1A9,'AL']])
call extend(s:tmp, [[0x1D1AA,0x1D1AD,'CM'],[0x1D1AE,0x1D1E8,'AL'],[0x1D200,0x1D241,'AL']])
call extend(s:tmp, [[0x1D242,0x1D244,'CM'],[0x1D245,0x1D245,'AL'],[0x1D300,0x1D356,'AL']])
call extend(s:tmp, [[0x1D360,0x1D371,'AL'],[0x1D400,0x1D454,'AL'],[0x1D456,0x1D49C,'AL']])
call extend(s:tmp, [[0x1D49E,0x1D49F,'AL'],[0x1D4A2,0x1D4A2,'AL'],[0x1D4A5,0x1D4A6,'AL']])
call extend(s:tmp, [[0x1D4A9,0x1D4AC,'AL'],[0x1D4AE,0x1D4B9,'AL'],[0x1D4BB,0x1D4BB,'AL']])
call extend(s:tmp, [[0x1D4BD,0x1D4C3,'AL'],[0x1D4C5,0x1D505,'AL'],[0x1D507,0x1D50A,'AL']])
call extend(s:tmp, [[0x1D50D,0x1D514,'AL'],[0x1D516,0x1D51C,'AL'],[0x1D51E,0x1D539,'AL']])
call extend(s:tmp, [[0x1D53B,0x1D53E,'AL'],[0x1D540,0x1D544,'AL'],[0x1D546,0x1D546,'AL']])
call extend(s:tmp, [[0x1D54A,0x1D550,'AL'],[0x1D552,0x1D6A5,'AL'],[0x1D6A8,0x1D7CB,'AL']])
call extend(s:tmp, [[0x1D7CE,0x1D7FF,'NU'],[0x1D800,0x1D9FF,'AL'],[0x1DA00,0x1DA36,'CM']])
call extend(s:tmp, [[0x1DA37,0x1DA3A,'AL'],[0x1DA3B,0x1DA6C,'CM'],[0x1DA6D,0x1DA74,'AL']])
call extend(s:tmp, [[0x1DA75,0x1DA75,'CM'],[0x1DA76,0x1DA83,'AL'],[0x1DA84,0x1DA84,'CM']])
call extend(s:tmp, [[0x1DA85,0x1DA86,'AL'],[0x1DA87,0x1DA8A,'BA'],[0x1DA8B,0x1DA8B,'AL']])
call extend(s:tmp, [[0x1DA9B,0x1DA9F,'CM'],[0x1DAA1,0x1DAAF,'CM'],[0x1E000,0x1E006,'CM']])
call extend(s:tmp, [[0x1E008,0x1E018,'CM'],[0x1E01B,0x1E021,'CM'],[0x1E023,0x1E024,'CM']])
call extend(s:tmp, [[0x1E026,0x1E02A,'CM'],[0x1E800,0x1E8C4,'AL'],[0x1E8C7,0x1E8CF,'AL']])
call extend(s:tmp, [[0x1E8D0,0x1E8D6,'CM'],[0x1E900,0x1E943,'AL'],[0x1E944,0x1E94A,'CM']])
call extend(s:tmp, [[0x1E950,0x1E959,'NU'],[0x1E95E,0x1E95F,'OP'],[0x1EE00,0x1EE03,'AL']])
call extend(s:tmp, [[0x1EE05,0x1EE1F,'AL'],[0x1EE21,0x1EE22,'AL'],[0x1EE24,0x1EE24,'AL']])
call extend(s:tmp, [[0x1EE27,0x1EE27,'AL'],[0x1EE29,0x1EE32,'AL'],[0x1EE34,0x1EE37,'AL']])
call extend(s:tmp, [[0x1EE39,0x1EE39,'AL'],[0x1EE3B,0x1EE3B,'AL'],[0x1EE42,0x1EE42,'AL']])
call extend(s:tmp, [[0x1EE47,0x1EE47,'AL'],[0x1EE49,0x1EE49,'AL'],[0x1EE4B,0x1EE4B,'AL']])
call extend(s:tmp, [[0x1EE4D,0x1EE4F,'AL'],[0x1EE51,0x1EE52,'AL'],[0x1EE54,0x1EE54,'AL']])
call extend(s:tmp, [[0x1EE57,0x1EE57,'AL'],[0x1EE59,0x1EE59,'AL'],[0x1EE5B,0x1EE5B,'AL']])
call extend(s:tmp, [[0x1EE5D,0x1EE5D,'AL'],[0x1EE5F,0x1EE5F,'AL'],[0x1EE61,0x1EE62,'AL']])
call extend(s:tmp, [[0x1EE64,0x1EE64,'AL'],[0x1EE67,0x1EE6A,'AL'],[0x1EE6C,0x1EE72,'AL']])
call extend(s:tmp, [[0x1EE74,0x1EE77,'AL'],[0x1EE79,0x1EE7C,'AL'],[0x1EE7E,0x1EE7E,'AL']])
call extend(s:tmp, [[0x1EE80,0x1EE89,'AL'],[0x1EE8B,0x1EE9B,'AL'],[0x1EEA1,0x1EEA3,'AL']])
call extend(s:tmp, [[0x1EEA5,0x1EEA9,'AL'],[0x1EEAB,0x1EEBB,'AL'],[0x1EEF0,0x1EEF1,'AL']])
call extend(s:tmp, [[0x1F000,0x1F0FF,'ID'],[0x1F100,0x1F10C,'AI'],[0x1F10D,0x1F10F,'ID']])
call extend(s:tmp, [[0x1F110,0x1F12D,'AI'],[0x1F12E,0x1F12E,'AL'],[0x1F12F,0x1F12F,'ID']])
call extend(s:tmp, [[0x1F130,0x1F169,'AI'],[0x1F16A,0x1F16B,'AL'],[0x1F16C,0x1F16F,'ID']])
call extend(s:tmp, [[0x1F170,0x1F1AC,'AI'],[0x1F1AD,0x1F1E5,'ID'],[0x1F1E6,0x1F1FF,'RI']])
call extend(s:tmp, [[0x1F200,0x1F384,'ID'],[0x1F385,0x1F385,'EB'],[0x1F386,0x1F39B,'ID']])
call extend(s:tmp, [[0x1F39C,0x1F39D,'AL'],[0x1F39E,0x1F3B4,'ID'],[0x1F3B5,0x1F3B6,'AL']])
call extend(s:tmp, [[0x1F3B7,0x1F3BB,'ID'],[0x1F3BC,0x1F3BC,'AL'],[0x1F3BD,0x1F3C2,'ID']])
call extend(s:tmp, [[0x1F3C3,0x1F3C4,'EB'],[0x1F3C5,0x1F3C9,'ID'],[0x1F3CA,0x1F3CB,'EB']])
call extend(s:tmp, [[0x1F3CC,0x1F3FA,'ID'],[0x1F3FB,0x1F3FF,'EM'],[0x1F400,0x1F441,'ID']])
call extend(s:tmp, [[0x1F442,0x1F443,'EB'],[0x1F444,0x1F445,'ID'],[0x1F446,0x1F450,'EB']])
call extend(s:tmp, [[0x1F451,0x1F465,'ID'],[0x1F466,0x1F469,'EB'],[0x1F46A,0x1F46D,'ID']])
call extend(s:tmp, [[0x1F46E,0x1F46E,'EB'],[0x1F46F,0x1F46F,'ID'],[0x1F470,0x1F478,'EB']])
call extend(s:tmp, [[0x1F479,0x1F47B,'ID'],[0x1F47C,0x1F47C,'EB'],[0x1F47D,0x1F480,'ID']])
call extend(s:tmp, [[0x1F481,0x1F483,'EB'],[0x1F484,0x1F484,'ID'],[0x1F485,0x1F487,'EB']])
call extend(s:tmp, [[0x1F488,0x1F49F,'ID'],[0x1F4A0,0x1F4A0,'AL'],[0x1F4A1,0x1F4A1,'ID']])
call extend(s:tmp, [[0x1F4A2,0x1F4A2,'AL'],[0x1F4A3,0x1F4A3,'ID'],[0x1F4A4,0x1F4A4,'AL']])
call extend(s:tmp, [[0x1F4A5,0x1F4A9,'ID'],[0x1F4AA,0x1F4AA,'EB'],[0x1F4AB,0x1F4AE,'ID']])
call extend(s:tmp, [[0x1F4AF,0x1F4AF,'AL'],[0x1F4B0,0x1F4B0,'ID'],[0x1F4B1,0x1F4B2,'AL']])
call extend(s:tmp, [[0x1F4B3,0x1F4FF,'ID'],[0x1F500,0x1F506,'AL'],[0x1F507,0x1F516,'ID']])
call extend(s:tmp, [[0x1F517,0x1F524,'AL'],[0x1F525,0x1F531,'ID'],[0x1F532,0x1F549,'AL']])
call extend(s:tmp, [[0x1F54A,0x1F574,'ID'],[0x1F575,0x1F575,'EB'],[0x1F576,0x1F579,'ID']])
call extend(s:tmp, [[0x1F57A,0x1F57A,'EB'],[0x1F57B,0x1F58F,'ID'],[0x1F590,0x1F590,'EB']])
call extend(s:tmp, [[0x1F591,0x1F594,'ID'],[0x1F595,0x1F596,'EB'],[0x1F597,0x1F5D3,'ID']])
call extend(s:tmp, [[0x1F5D4,0x1F5DB,'AL'],[0x1F5DC,0x1F5F3,'ID'],[0x1F5F4,0x1F5F9,'AL']])
call extend(s:tmp, [[0x1F5FA,0x1F644,'ID'],[0x1F645,0x1F647,'EB'],[0x1F648,0x1F64A,'ID']])
call extend(s:tmp, [[0x1F64B,0x1F64F,'EB'],[0x1F650,0x1F675,'AL'],[0x1F676,0x1F678,'QU']])
call extend(s:tmp, [[0x1F679,0x1F67B,'NS'],[0x1F67C,0x1F67F,'AL'],[0x1F680,0x1F6A2,'ID']])
call extend(s:tmp, [[0x1F6A3,0x1F6A3,'EB'],[0x1F6A4,0x1F6B3,'ID'],[0x1F6B4,0x1F6B6,'EB']])
call extend(s:tmp, [[0x1F6B7,0x1F6BF,'ID'],[0x1F6C0,0x1F6C0,'EB'],[0x1F6C1,0x1F6FF,'ID']])
call extend(s:tmp, [[0x1F700,0x1F773,'AL'],[0x1F774,0x1F77F,'ID'],[0x1F780,0x1F7D4,'AL']])
call extend(s:tmp, [[0x1F7D5,0x1F7FF,'ID'],[0x1F800,0x1F80B,'AL'],[0x1F80C,0x1F80F,'ID']])
call extend(s:tmp, [[0x1F810,0x1F847,'AL'],[0x1F848,0x1F84F,'ID'],[0x1F850,0x1F859,'AL']])
call extend(s:tmp, [[0x1F85A,0x1F85F,'ID'],[0x1F860,0x1F887,'AL'],[0x1F888,0x1F88F,'ID']])
call extend(s:tmp, [[0x1F890,0x1F8AD,'AL'],[0x1F8AE,0x1F917,'ID'],[0x1F918,0x1F91E,'EB']])
call extend(s:tmp, [[0x1F91F,0x1F925,'ID'],[0x1F926,0x1F926,'EB'],[0x1F927,0x1F92F,'ID']])
call extend(s:tmp, [[0x1F930,0x1F930,'EB'],[0x1F931,0x1F932,'ID'],[0x1F933,0x1F939,'EB']])
call extend(s:tmp, [[0x1F93A,0x1F93B,'ID'],[0x1F93C,0x1F93E,'EB'],[0x1F93F,0x1FFFD,'ID']])
call extend(s:tmp, [[0x20000,0x2FFFD,'ID'],[0x30000,0x3FFFD,'ID'],[0xE0001,0xE0001,'CM']])
call extend(s:tmp, [[0xE0020,0xE007F,'CM'],[0xE0100,0xE01EF,'CM'],[0xF0000,0xFFFFD,'XX']])
call extend(s:tmp, [[0x100000,0x10FFFD,'XX']])
let s:lib.linebreak_table = s:tmp
unlet s:tmp

let s:tmp = []
call add(s:tmp, ['CM','CM','CM','CM','CM','CM','CM','CM','CM','BA','LF','BK','BK','CR','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','SP','EX','QU','AL','PR','PO','AL','QU','OP','CP','AL','PR','IS','HY','IS','SY','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','IS','IS','AL','AL','AL','EX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','PR','CP','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','BA','CL','AL','CM','CM','CM','CM','CM','CM','NL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','GL','OP','PO','PR','PR','PR','AL','AI','AI','AL','AI','QU','AL','BA','AL','AL','PO','PR','AI','AI','BB','AL','AI','AI','AI','AI','AI','QU','AI','AI','AI','OP','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','BB','AI','AI','AI','BB','AI','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AI','AI','AI','AI','AL','AI','AL','BB','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','GL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','GL','GL','GL','GL','GL','GL','GL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','IS','AL','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','IS','BA','XX','XX','AL','AL','PR','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','BA','CM','AL','CM','CM','AL','CM','CM','EX','CM','XX','XX','XX','XX','XX','XX','XX','XX','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','XX','XX','XX','XX','XX','HL','HL','HL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','PO','PO','PO','IS','IS','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','EX','CM','XX','EX','EX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','PO','NU','NU','AL','AL','AL','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','EX','AL','CM','CM','CM','CM','CM','CM','CM','AL','AL','CM','CM','CM','CM','CM','CM','AL','AL','CM','CM','AL','CM','CM','CM','CM','AL','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','IS','EX','AL','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','CM','CM','CM','AL','CM','CM','CM','CM','CM','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','XX','XX','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM'])
call add(s:tmp, ['CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','BA','BA','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','XX','XX','AL','AL','AL','AL','XX','XX','CM','AL','CM','CM','CM','CM','CM','CM','CM','XX','XX','CM','CM','XX','XX','CM','CM','CM','AL','XX','XX','XX','XX','XX','XX','XX','XX','CM','XX','XX','XX','XX','AL','AL','XX','AL','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','PO','PO','AL','AL','AL','AL','AL','PO','AL','PR','XX','XX','XX','XX'])
call add(s:tmp, ['XX','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','XX','AL','AL','XX','AL','AL','XX','XX','CM','XX','CM','CM','CM','CM','CM','XX','XX','XX','XX','CM','CM','XX','XX','CM','CM','CM','XX','XX','XX','CM','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','XX','AL','XX','XX','XX','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','CM','CM','AL','AL','AL','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','XX','AL','AL','AL','AL','AL','XX','XX','CM','AL','CM','CM','CM','CM','CM','CM','CM','CM','XX','CM','CM','CM','XX','CM','CM','CM','XX','XX','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','PR','XX','XX','XX','XX','XX','XX','XX','AL','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['XX','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','XX','AL','AL','AL','AL','AL','XX','XX','CM','AL','CM','CM','CM','CM','CM','CM','CM','XX','XX','CM','CM','XX','XX','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','CM','CM','XX','XX','XX','XX','AL','AL','XX','AL','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','CM','AL','XX','AL','AL','AL','AL','AL','AL','XX','XX','XX','AL','AL','AL','XX','AL','AL','AL','AL','XX','XX','XX','AL','AL','XX','AL','XX','AL','AL','XX','XX','XX','AL','AL','XX','XX','XX','AL','AL','AL','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','CM','CM','CM','CM','CM','XX','XX','XX','CM','CM','CM','XX','CM','CM','CM','CM','XX','XX','AL','XX','XX','XX','XX','XX','XX','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','PR','AL','XX','XX','XX','XX','XX'])
call add(s:tmp, ['CM','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','AL','CM','CM','CM','CM','CM','CM','CM','XX','CM','CM','CM','XX','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','CM','CM','XX','AL','AL','AL','XX','XX','XX','XX','XX','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','XX','XX','CM','AL','CM','CM','CM','CM','CM','CM','CM','XX','CM','CM','CM','XX','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','CM','CM','XX','XX','XX','XX','XX','XX','XX','AL','XX','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['XX','CM','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','CM','CM','CM','CM','CM','CM','CM','XX','CM','CM','CM','XX','CM','CM','CM','CM','AL','AL','XX','XX','XX','XX','AL','AL','AL','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','PO','AL','AL','AL','AL','AL','AL','XX','XX','CM','CM','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','CM','XX','XX','XX','XX','CM','CM','CM','CM','CM','CM','XX','CM','XX','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','CM','CM','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','XX','XX','PR','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','BA','BA','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','SA','SA','XX','SA','XX','XX','SA','SA','XX','SA','XX','XX','SA','XX','XX','XX','XX','XX','XX','SA','SA','SA','SA','XX','SA','SA','SA','SA','SA','SA','SA','XX','SA','SA','SA','XX','SA','XX','SA','XX','XX','SA','SA','XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','SA','SA','SA','XX','XX','SA','SA','SA','SA','SA','XX','SA','XX','SA','SA','SA','SA','SA','SA','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','SA','SA','SA','SA','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','BB','BB','BB','BB','AL','BB','BB','GL','BB','BB','BA','GL','EX','EX','EX','EX','EX','GL','AL','EX','AL','AL','AL','CM','CM','AL','AL','AL','AL','AL','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','CM','AL','CM','AL','CM','OP','CL','OP','CL','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','BA','CM','CM','CM','CM','CM','BA','CM','CM','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','BA','BA','AL','AL','AL','AL','AL','AL','CM','AL','AL','AL','AL','AL','AL','XX','AL','AL','BB','BB','BA','BB','AL','AL','AL','AL','AL','GL','GL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','BA','BA','AL','AL','AL','AL','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','SA','SA','SA','SA','SA','SA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','XX','XX','XX','XX','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','CM','CM','CM','AL','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','XX','XX'])
call add(s:tmp, ['BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','BA','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','BA','BA','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','XX','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','BA','BA','NS','SA','BA','AL','BA','PR','SA','SA','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','EX','EX','BA','BA','BB','AL','EX','EX','AL','CM','CM','CM','GL','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','AL','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','AL','XX','XX','XX','EX','EX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','SA','SA','SA','SA','SA','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','XX','XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','SA','XX','XX','XX','SA','SA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','XX','XX','AL','AL','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','CM','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','BA','BA','AL','BA','BA','BA','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','BA','BA','BA','BA','BA','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','AL','AL','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','CM','CM','CM','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','CM','AL','AL','AL','AL','CM','CM','CM','AL','AL','XX','CM','CM','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','CM','CM','CM','CM','CM'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','AL','XX','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','BB','AL','XX'])
call add(s:tmp, ['BA','BA','BA','BA','BA','BA','BA','GL','BA','BA','BA','ZW','CM','ZWJ','CM','CM','BA','GL','BA','BA','B2','AI','AI','AL','QU','QU','OP','QU','QU','QU','OP','QU','AI','AI','AL','AL','IN','IN','IN','BA','BK','BK','CM','CM','CM','CM','CM','GL','PO','PO','PO','PO','PO','PO','PO','PO','AL','QU','QU','AI','NS','NS','AL','AL','AL','AL','AL','AL','IS','OP','CL','NS','NS','NS','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','AL','BA','BA','BA','BA','AL','BA','BA','BA','WJ','AL','AL','AL','AL','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','XX','XX','AI','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','AI','AL','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','PR','PR','PR','PR','PR','PR','PR','PO','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PO','PR','PR','PR','PR','PO','PR','PR','PO','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','PR','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','PO','AL','AI','AL','AL','AL','PO','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','PR','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AL','AL','AL','AL','AI','AL','AL','AI','AL','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','XX','XX','XX','XX','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AI','AL','AI','AI','AL','AL','AL','AI','AI','AL','AL','AI','AL','AL','AL','AI','AL','AI','PR','PR','AL','AI','AL','AL','AL','AL','AI','AL','AL','AI','AI','AI','AI','AL','AL','AI','AL','AI','AL','AI','AI','AI','AI','AI','AI','AL','AI','AL','AL','AL','AL','AL','AI','AI','AI','AI','AL','AL','AL','AL','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AI','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AL','AI','AI','AI','AI','AL','AL','AI','AI','AL','AL','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AL','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','IN','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','OP','CL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','ID','ID','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','ID','ID','ID','ID','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL'])
call add(s:tmp, ['AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AL','AI','AI','AL','AL','AL','AL','AI','AI','AL','AL','AI','AI','AL','AL','AL','AL','AI','AI','AI','AL','AL','AI','AL','AL','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['ID','ID','ID','ID','AL','AI','AI','AL','AL','AI','AL','AL','AL','AL','AI','AI','AL','AL','AL','AL','ID','ID','AI','AI','ID','AL','ID','ID','ID','EB','ID','ID','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','ID','ID','ID','AL','AL','AL','AL','AI','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AI','AI','AI','AL','AI','ID','AI','AI','AL','AI','AI','AL','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','ID','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','AI','AI','AI','AI','ID','AL','ID','ID','ID','AI','ID','ID','AI','AI','AI','ID','ID','AI','AI','ID','AI','AI','ID','ID','ID','AL','AI','AL','AL','AL','AL','AI','AI','ID','AI','AI','AI','AI','AI','AI','ID','ID','ID','ID','ID','AI','ID','ID','EB','ID','AI','AI','ID','ID','ID'])
call add(s:tmp, ['ID','ID','ID','ID','ID','AL','AL','AL','ID','ID','EB','EB','EB','EB','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AL','AL','AL','QU','QU','QU','QU','QU','QU','AL','EX','EX','ID','AL','AL','AL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','OP','CL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','OP','CL','AL','AL'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AI','AI','AI','AI','AI','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','AL','AL','XX','XX','XX','XX','XX','EX','BA','BA','BA','AL','EX','BA'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','XX','XX','XX','XX','XX','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','AL','BA','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM'])
call add(s:tmp, ['QU','QU','QU','QU','QU','QU','QU','QU','QU','QU','QU','QU','QU','QU','BA','BA','BA','BA','BA','BA','BA','BA','AL','BA','OP','BA','AL','AL','QU','QU','AL','AL','QU','QU','OP','CL','OP','CL','OP','CL','OP','CL','BA','BA','BA','BA','EX','AL','BA','BA','AL','BA','BA','AL','AL','AL','AL','AL','B2','B2','BA','BA','BA','AL','BA','BA','OP','BA','BA','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','XX'])
call add(s:tmp, ['BA','CL','CL','ID','ID','NS','ID','ID','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','ID','ID','OP','CL','OP','CL','OP','CL','OP','CL','NS','OP','CL','CL','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','CM','CM','CM','CM','CM','CM','ID','ID','ID','ID','ID','CM','ID','ID','ID','ID','ID','NS','NS','ID','ID','ID','XX','CJ','ID','CJ','ID','CJ','ID','CJ','ID','CJ','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','CJ','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','CJ','ID','CJ','ID','CJ','ID','ID','ID','ID','ID','ID','CJ','ID','ID','ID','ID','ID','ID','CJ','CJ','XX','XX','CM','CM','NS','NS','NS','NS','ID','NS','CJ','ID','CJ','ID','CJ','ID','CJ','ID','CJ','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','CJ','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','CJ','ID','CJ','ID','CJ','ID','ID','ID','ID','ID','ID','CJ','ID','ID','ID','ID','ID','ID','CJ','CJ','ID','ID','ID','ID','NS','CJ','NS','NS','ID'])
call add(s:tmp, ['XX','XX','XX','XX','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','XX','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ'])
call add(s:tmp, ['ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','AI','AI','AI','AI','AI','AI','AI','AI','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','NS','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','BA'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BA','EX','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','AL','BA','BA','BA','BA','BA','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL','AL','CM','AL','AL','AL','CM','AL','AL','AL','AL','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','AL','AL','AL','AL','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','PO','AL','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BB','BB','EX','EX','XX','XX','XX','XX','XX','XX','XX','XX','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','BA','BA','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','BB','AL','XX','XX'])
call add(s:tmp, ['NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','BA','BA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','JL','XX','XX','XX','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','AL','AL','AL','AL','AL','AL','BA','BA','BA','AL','AL','AL','AL','XX','AL','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','AL','AL','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','SA','SA','SA','SA','SA','XX'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','CM','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','AL','BA','BA','BA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','SA','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','SA','SA','SA','SA','SA','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','BA','BA','AL','AL','AL','CM','CM','XX','XX','XX','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['XX','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CM','CM','CM','CM','CM','CM','CM','CM','BA','CM','CM','XX','XX','NU','NU','NU','NU','NU','NU','NU','NU','NU','NU','XX','XX','XX','XX','XX','XX'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3'])
call add(s:tmp, ['H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3'])
call add(s:tmp, ['H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H2','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','H3','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','JV','XX','XX','XX','XX','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','JT','XX','XX','XX','XX'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['SG'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['XX'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['ID'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','HL','CM','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','AL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','XX','HL','HL','HL','HL','HL','XX','HL','XX','HL','HL','XX','HL','HL','XX','HL','HL','HL','HL','HL','HL','HL','HL','HL','HL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL'])
call add(s:tmp, ['AL'])
call add(s:tmp, ['AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','CL','OP','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','PO','AL','XX','XX'])
call add(s:tmp, ['CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','IS','CL','CL','IS','IS','EX','EX','OP','CL','IN','XX','XX','XX','XX','XX','XX','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','CM','ID','ID','ID','ID','ID','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','OP','CL','ID','ID','OP','CL','ID','ID','ID','ID','ID','ID','ID','CL','ID','CL','XX','NS','NS','EX','EX','ID','OP','CL','OP','CL','OP','CL','ID','ID','ID','ID','ID','ID','ID','ID','XX','ID','PR','PO','ID','XX','XX','XX','XX','AL','AL','AL','AL','AL','XX','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','AL','XX','XX','WJ'])
call add(s:tmp, ['XX','EX','ID','ID','PR','PO','ID','ID','OP','CL','ID','ID','CL','ID','CL','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','NS','NS','ID','ID','ID','EX','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','OP','ID','CL','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','OP','ID','CL','ID','OP','CL','CL','OP','CL','CL','NS','ID','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','CJ','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','NS','NS','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','ID','XX','XX','XX','ID','ID','ID','ID','ID','ID','XX','XX','ID','ID','ID','ID','ID','ID','XX','XX','ID','ID','ID','ID','ID','ID','XX','XX','ID','ID','ID','XX','XX','XX','PO','PR','ID','ID','ID','PR','PR','XX','AL','AL','AL','AL','AL','AL','AL','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','CM','CM','CM','CB','AI','XX','XX'])
let s:lib.linebreak_bmp = s:tmp
unlet s:tmp

let &cpo = s:cpo_save

