" vi:set ts=8 sts=2 sw=2 tw=0:
"
" autodate.vim - A plugin to update time stamps automatically
"
" Maintainer:	MURAOKA Taro <koron.kaoriya@gmail.com>
" Last Change:	06-Feb-2006.

" Description:
" Command:
"   :Autodate	    Manually autodate.
"   :AutodateON	    Turn on autodate in current buffer (default).
"   :AutodateOFF    Turn off autodate in current buffer.
"
" Options:
"   Each global variable (option) is overruled by buffer variable (what
"   starts with "b:").
"
"   'autodate_format'
"	Format string used for time stamps.  See |strftime()| for details.
"	See MonthnameString() for special extension of format.
"	Default: '%d-%3m-%Y'
"
"   'autodate_lines'
"	The number of lines searched for the existence of a time stamp when
"	writing a buffer.  The search range will be from top of buffer (or
"	line 'autodate_start_line') to 'autodate_lines' lines below.  The
"	bigger value you have, the longer it'll take to search words.  You
"	can expect to improve performance by decreasing the value of this
"	option.
"	Default: 50
"
"   'autodate_start_line'
"	Line number to start searching for time stamps when writing buffer
"	to file.
"	If minus, line number is counted from the end of file.
"	Default: 1
"
"   'autodate_keyword_pre'
"	A prefix pattern (see |pattern| for details) which denotes time
"	stamp's location.  If empty, default value will be used.
"	Default: '\cLast Change:'
"
"   'autodate_keyword_post'
"	A postfix pattern which denotes time stamp's location.  If empty,
"	default value will be used.
"	Default: '\.'
"
" Usage:
"   Write a line as below (case ignored) somewhere in the first 50 lines of
"   a buffer:
"	Last Change: .
"   When writing the buffer to a file, this line will be modified and a time
"   stamp will be inserted automatically.  Example:
"	Last Change: 11-May-2002.
"
"   You can execute :Autodate command to update time stamps manually.  The
"   range of lines which looks for a time stamp can also be specified.  When
"   no range is given, the command will be applied to the current line.
"   Example:
"	:%Autodate		" Whole file
"	:\<,\>Autodate		" Range selected by visual mode
"	:Autodate		" Current cursor line
"
"   The format of the time stamp to insert can be specified by
"   'autodate_format' option.  See |strftime()| (vim script function) for
"   details.  Sample format settings and the corresponding outputs are show
"   below.
"	FORMAT: %Y/%m/%d	OUTPUT: 2001/12/24
"	FORMAT: %H:%M:%S	OUTPUT: 10:06:32
"	FORMAT: %y%m%d-%H%M	OUTPUT: 011224-1006
"
"   Autodate.vim determines where to insert a time stamp by looking for a
"   KEYWORD.  A keyword consists of a PREFIX and a POSTFIX part, and they
"   can be set by 'autodate_keyword_pre' and 'autodate_keyword_post'
"   options, respectively.  If you set these values as below in your .vimrc:
"	:let autodate_format = ': %Y/%m/%d %H:%M:%S '
"	:let autodate_keyword_pre  = '\$Date'
"	:let autodate_keyword_post = '\$'
"   They will function like $Date$ in cvs.  Example:
"	$Date: 2001/12/24 10:06:32 $
"
"   Just another application. To insert a time stamp between '<!--DATE-->'
"   when writing HTML, try below:
"	:let b:autodate_keyword_pre = '<!--DATE-->'
"	:let b:autodate_keyword_post = '<!--DATE-->'
"   It will be useful if to put these lines in your ftplugin/html.vim.
"   Example:
"	<!--DATE-->24-Dec-2001<!--DATE-->
"
"   In addition, priority is given to a buffer local option (what starts in
"   b:) about all the options of autodate.
"
"
" To make vim NOT TO LOAD this plugin, write next line in your .vimrc:
"	:let plugin_autodate_disable = 1

" Japanese Description:
" R}h:
"   :Autodate	    蓮Ń^CX^vXV
"   :AutodateON	    ݂̃obt@̎XVL
"   :AutodateOFF    ݂̃obt@̎XV𖳌
"
" IvV: (ꂼ̃IvV̓obt@(b:)D悳)
"
"   'autodate_format'
"	^CX^vɎgptH[}bgBtH[}bg̏ڍׂ
"	|strftime()|QƁBtH[}bgւ̓ƎgɂĂMonthnameString()
"	QƁBȗl: '%d-%3m-%Y'
"
"   'autodate_lines'
"	ۑɃ^CX^v݂̑`FbNsB₹Α₷ق
"	L[[h邽߂ɎԂ蓮삪xȂBtɒxƊ
"	Ƃɂ͏Ȓlݒ肷΃ptH[}X̉P҂łB
"	ȗl: 50
"
"   'autodate_keyword_pre'
"	^CX^v݂̑OuL[[h(K\)BK{B󕶎
"	w肷ƏȗlgBȗl: '\cLast Change:'
"
"   'autodate_keyword_post'
"	^CX^v݂̑uL[[h(K\)BK{B󕶎
"	w肷ƏȗlgBȗl: '\.'
"
" gp@:
"   t@C̐擪50sȓ
"	Last Change: .
"   Əs(啶͋ʂ܂)pӂƁAt@C̕ۑ(:w)
"   ɎIɂ̎(^CX^v)}܂Bʗ:
"	Last Change: 11-May-2002.
"
"   ExR}h:Autodates邱ƂŎ蓮Ń^CX^v̍XVsȂ
"   ܂B̍ۂɃ^CX^vT͈͂w肷邱Ƃł܂Bɔ͈
"   w肵Ȃ΃J[\̂sΏۂɂȂ܂B:
"	:%Autodate		" t@CS
"	:\<,\>Autodate		" rWAÏ
"	:Autodate		" ݃J[\̂s
"
"   }^CX^v̏̓IvV'autodate_format'Ŏw肷邱
"   ł܂BڍׂVimXNvg֐|strftime()|̐ɏ]܂Bȉ
"   Ƃ̏o̗͂܂:
"	: %Y/%m/%d		o: 2001/12/24
"	: %H:%M:%S		o: 10:06:32
"	: %y%m%d-%H%M	o: 011224-1006
"
"   autodate.vim̓L[[hTƂŃ^CX^v}ׂʒu
"   Ă܂BL[[h͑OuƌuȂAꂼIvV
"   'autodate_keyword_pre''autodate_keyword_post'ݒ肷邱ƂŕύXł
"   ܂Blݒt@C(_vimrc)Ŏ̂悤ɐݒ肷:
"	:let autodate_format = ': %Y/%m/%d %H:%M:%S '
"	:let autodate_keyword_pre  = '\$Date'
"	:let autodate_keyword_post = '\$'
"   cvsɂ$Date$̂悤ɓ삵܂B:
"	$Date: 2001/12/24 10:06:32 $
"
"   pƂHTMLLqۂ<!--DATE-->ň͂܂ꂽɃ^CX^v}
"   ꍇɂ:
"	:let b:autodate_keyword_pre = '<!--DATE-->'
"	:let b:autodate_keyword_post = '<!--DATE-->'
"   Ǝw肵܂Bftplugin/html.vimŐݒ肷ƕ֗ł傤B:
"	<!--DATE-->24-Dec-2001<!--DATE-->
"
"   ȂautodatȇẴIvVɂāAobt@[JIvV(b:
"   n܂)D悳܂B
"
" ̃vOCǍ݂Ȃ.vimrcɎ̂悤ɏ:
"	:let plugin_autodate_disable = 1

if exists('plugin_autodate_disable')
  finish
endif
let s:debug = 0

"---------------------------------------------------------------------------
"				    Options

"
" 'autodate_format'
"
if !exists('autodate_format')
  let g:autodate_format = '%d-%3m-%Y'
endif

"
" 'autodate_lines'
"
if !exists('autodate_lines')
  let g:autodate_lines = 50
endif

"
" 'autodate_start_line'
"
if !exists('autodate_start_line')
  let g:autodate_start_line = 1
endif

"
" 'autodate_keyword_pre'
"
if !exists('autodate_keyword_pre')
  let g:autodate_keyword_pre = '\cLast Change:'
endif

"
" 'autodate_keyword_post'
"
if !exists('autodate_keyword_post')
  let g:autodate_keyword_post = '\.'
endif

"---------------------------------------------------------------------------
"				    Mappings

command! -range Autodate call <SID>Autodate(<line1>, <line2>)
command! AutodateOFF let b:autodate_disable = 1
command! AutodateON let b:autodate_disable = 0
if has("autocmd")
  augroup Autodate
    au!
    autocmd BufUnload,FileWritePre,BufWritePre * call <SID>Autodate()
  augroup END
endif " has("autocmd")

"---------------------------------------------------------------------------
"				 Implementation

"
" Autodate([{firstline} [, {lastline}]])
"
"   {firstline}{lastline}Ŏw肳ꂽ͈͂ɂāA^CX^v̎
"   Abvf[gsȂB{firstline}ȗꂽꍇ̓t@C̐擪A
"   {lastline}ȗꂽꍇ{firstline}'autodate_lines's͈̔͂
"   ۂɂȂB
"
function! s:Autodate(...)
  " Check enable
  if (exists('b:autodate_disable') && b:autodate_disable != 0) || &modified == 0
    return
  endif

  " Verify {firstline}
  if a:0 > 0 && a:1 > 0
    let firstline = a:1
  else
    let firstline = s:GetAutodateStartLine()
  endif

  " Verify {lastline}
  if a:0 > 1 && a:2 <= line('$')
    let lastline = a:2
  else
    let lastline = firstline + s:GetAutodateLines() - 1
    " Range check
    if lastline > line('$')
      let lastline = line('$')
    endif
  endif

  if firstline <= lastline
    call s:AutodateStub(firstline, lastline)
  endif
endfunction

"
" GetAutodateStartLine()
"
"   TJn_擾
"
function! s:GetAutodateStartLine()
  let retval = 1
  if exists('b:autodate_start_line')
    let retval = b:autodate_start_line
  elseif exists('g:autodate_start_line')
    let retval = g:autodate_start_line
  endif

  if retval < 0
    let retval = retval + line('$') + 1
  endif
  if retval <= 0
    let retval = 1
  endif
  return retval
endfunction

"
" GetAutodateLines()
"   
"   autodateΏ۔͈͂擾
"
function! s:GetAutodateLines()
  if exists('b:autodate_lines') && b:autodate_lines > 0
    return b:autodate_lines
  elseif exists('g:autodate_lines') && g:autodate_lines > 0
    return g:autodate_lines
  else
    return 50
  endif
endfunction

"
" AutodateStub(first, last)
"
"   w肳ꂽ͈͂ɂă^CX^v̎Abvf[gsȂB
"
function! s:AutodateStub(first, last)

  " Verify pre-keyword.
  if exists('b:autodate_keyword_pre') && b:autodate_keyword_pre != ''
    let pre = b:autodate_keyword_pre
  else
    if exists('g:autodate_keyword_pre') && g:autodate_keyword_pre != ''
      let pre = g:autodate_keyword_pre
    else
      let pre = '\cLast Change:'
    endif
  endif

  " Verify post-keyword.
  if exists('b:autodate_keyword_post') && b:autodate_keyword_post != ''
    let post = b:autodate_keyword_post
  else
    if exists('g:autodate_keyword_post') && g:autodate_keyword_post != ''
      let post = g:autodate_keyword_post
    else
      let post = '\.'
    endif
  endif

  " Verify format.
  if exists('b:autodate_format') && b:autodate_format != ''
    let format = b:autodate_format
  else
    if exists('g:autodate_format') && g:autodate_format != ''
      let format = g:autodate_format
    else
      let format = '%d-%3m-%Y'
    endif
  endif

  " Generate substitution pattern
  let pat = '\('.pre.'\s*\)\(\S.*\)\?\('.post.'\)'
  let sub = Strftime2(format)
  " For debug
  if s:debug
    echo "range=".a:first."-".a:last
    echo "pat= ".pat
    echo "sub= ".sub
  endif

  " Process
  let i = a:first
  while i <= a:last
    let curline = getline(i)
    if curline =~ pat
      let newline = substitute(curline, pat, '\1' . sub . '\3', '')
      if curline !=# newline
	call setline(i, newline)
      endif
    endif
    let i = i + 1
  endwhile
endfunction

"
" Strftime2({format} [, {time}])
"   Enhanced version of strftime().
"
"   strftime()̃tH[}bggo[WBtH[}bg͂قƂǃIW
"   iƈꏏB̉pꖼɒuʂȏ %{n}m gp\B{n}
"   ͉pꖼ̒̕w肷(ő9)B0w肷Η]ȋ󔒂͕t
"   ȂB
"	:
"	    :echo Strftime2("%d-%3m-%Y")
"	    07-Oct-2001
"	    :echo Strftime2("%d-%0m-%Y")
"	    07-October-2001
"
function! Strftime2(...)
  if a:0 > 0
    " Get {time} argument.
    if a:0 > 1
      let time = a:2
    else
      let time = localtime()
    endif
    " Parse special format.
    let format = a:1
    let format = substitute(format, '%\(\d\+\)m', '\=MonthnameString(-1, submatch(1), time)', 'g')
    let format = substitute(format, '%\(\d\+\)a', '\=DaynameString(-1, submatch(1), time)', 'g')
    return strftime(format, time)
  endif
  " Genrate error!
  return strftime()
endfunction

"
" MonthnameString([{month} [, {length} [, {time}]]])
"   Get month name string in English with first specified letters.
"
"   p̌w肵ŕԂB{month}ȗꍇɂ݂͌̌
"   B{month}ɖȎw(1`12ȊO)sȂꂽꍇ{time}Ŏ
"   ԂB{time}ȗꍇɂ͑|localtime()|gp
"   B{length}ɂ͕ԂO̒w肷邪ȗƔCӒɂȂB
"	:
"	    :echo MonthnameString(8) . " 2001"
"	    August 2001
"	    :echo MonthnameString(8,3) . " 2001"
"	    Aug 2001
"
function! MonthnameString(...)
  " Get {time} argument.
  if a:0 > 2
    let time = a:3
  else
    let time = localtime()
  endif
  " Verify {month}.
  if a:0 > 0 && (a:1 >= 1 && a:1 <= 12)
    let month = a:1
  else
    let month = substitute(strftime('%m', time), '^0\+', '', '')
  endif
  " Verify {length}.
  if a:0 > 1 && (a:2 >= 1 && a:2 <= 9)
    let length = a:2
  else
    let length = strpart('785534469788', month - 1, 1)
  endif
  " Generate string of month name.
  return strpart('January  February March    April    May      June     July     August   SeptemberOctober  November December ', month * 9 - 9, length)
endfunction

"
" DaynameString([{month} [, {length} [, {time}]]])
"   Get day name string in English with first specified letters.
"
"   p̗jw肵ŕԂB{day}ȗꍇɂ͖{̗j
"   ԂB{day}ɖȎw(0`6ȊO)sȂꂽꍇ{time}Ŏ
"   jԂB{time}ȗꍇɂ͑|localtime()|gp
"   B{length}ɂ͕ԂO̒w肷邪ȗƔCӒɂȂB
"	:
"	    :echo DaynameString(0)
"	    Sunday
"	    :echo DaynameString(5,3).', 13th'
"	    Fri, 13th
"
function! DaynameString(...)
  " Get {time} argument.
  if a:0 > 2
    let time = a:3
  else
    let time = localtime()
  endif
  " Verify {day}.
  if a:0 > 0 && (a:1 >= 0 && a:1 <= 6)
    let day = a:1
  else
    let day = strftime('%w', time) + 0
  endif
  " Verify {length}.
  if a:0 > 1 && (a:2 >= 1 && a:2 <= 9)
    let length = a:2
  else
    let length = strpart('6798686', day, 1)
  endif
  " Generate string of day name.
  return strpart('Sunday   Monday   Tuesday  WednesdayThursday Friday   Saturday ', day * 9, length)
endfunction
