" vim:set ts=8 sts=2 sw=2 tw=0 nowrap:
"
" hz_ja.vim - Convert character between HANKAKU and ZENKAKU
"
" Last Change: 27-Aug-2015.
" Written By:  MURAOKA Taro <koron.kaoriya@gmail.com>
"
" Commands:
"   :Hankaku	  Convert to HANKAKU.
"   :Zenkaku	  Convert to ZENKAKU.
"   :ToggleHZ	  Toggle convert between HANKAKU and ZENKAKU.
"
" Functions:
"   ToHankaku(str)	Convert given string to HANKAKU.
"   ToZenkaku(str)	Convert given string to ZENKAKU.
"
" To make vim DO NOT LOAD this plugin, write next line in your .vimrc:
"	:let plugin_hz_ja_disable = 1

" Japanese Description:
" R}h:
"   :[range]Hankaku
"   :[range]Zenkaku
"   :[range]ToggleHZ
"   :[range]HzjaConvert {target}
"
" L[}bsO:
"   ȉ̓rWAÏɑ΂鑀łB܂()HzjaConvertR}h
"   Ɗ֐ɓnAtargetɎw\ȕłB
"     gHL	\ȕSĔpɕϊ	(han_all)
"     gZL	\ȕSđSpɕϊ	(zen_all)
"     gHA	ASCIISĔpɕϊ	(han_ascii)
"     gZA	ASCIISđSpɕϊ	(zen_ascii)
"     gHM	LSĔpɕϊ	(han_kigou)
"     gZM	LSđSpɕϊ	(zen_kigou)
"     gHW	pSĔpɕϊ	(han_eisu)
"     gZW	pSđSpɕϊ	(zen_eisu)
"     gHJ	J^JiSĔpɕϊ	(han_kana)
"     gZJ	J^JiSđSpɕϊ	(zen_kana)
"   ȉ͎gppx̍lāALƏd@\ƂĊ蓖Ăꂽ
"   L[}bvłB
"     gHH	ASCIISĔpɕϊ	(han_ascii)
"     gZZ	J^JiSđSpɕϊ	(zen_kana)
" 
" ֐:
"   ToHankaku(str)	𔼊p֕ϊ
"   ToZenkaku(str)	Sp֕ϊ
"   HzjaConvert(str, target)
"			targetɏ]ϊBtargeẗӖ̓L[}b
"			sO̍ڂQƁB
"
" j[g:
"   GUIł̓rWAĨ|bvAbvj[(ENbNj[)
"   ϊp̃R}hǉ܂B
"
" ̃vOCǍ݂Ȃ.vimrcɎ̂悤ɏ:
"	:let plugin_hz_ja_disable = 1

scriptencoding cp932

if exists('plugin_hz_ja_disable')
  finish
endif
if !has('multi_byte')
  finish
endif

command! -nargs=0 -range Hankaku <line1>,<line2>call <SID>ToggleLineWise('Hankaku')
command! -nargs=0 -range Zenkaku <line1>,<line2>call <SID>ToggleLineWise('Zenkaku')
command! -nargs=0 -range ToggleHZ <line1>,<line2>call <SID>ToggleLineWise('Toggle')
command! -nargs=1 -range -complete=custom,HzjaConvertComplete HzjaConvert <line1>,<line2>call <SID>HzjaConvert(<q-args>)

vnoremap <silent> gHL <C-\><C-N>:call <SID>HzjaConvertVisual('han_all')<CR>
vnoremap <silent> gZL <C-\><C-N>:call <SID>HzjaConvertVisual('zen_all')<CR>
vnoremap <silent> gHA <C-\><C-N>:call <SID>HzjaConvertVisual('han_ascii')<CR>
vnoremap <silent> gZA <C-\><C-N>:call <SID>HzjaConvertVisual('zen_ascii')<CR>
vnoremap <silent> gHM <C-\><C-N>:call <SID>HzjaConvertVisual('han_kigou')<CR>
vnoremap <silent> gZM <C-\><C-N>:call <SID>HzjaConvertVisual('zen_kigou')<CR>
vnoremap <silent> gHW <C-\><C-N>:call <SID>HzjaConvertVisual('han_eisu')<CR>
vnoremap <silent> gZW <C-\><C-N>:call <SID>HzjaConvertVisual('zen_eisu')<CR>
vnoremap <silent> gHJ <C-\><C-N>:call <SID>HzjaConvertVisual('han_kana')<CR>
vnoremap <silent> gZJ <C-\><C-N>:call <SID>HzjaConvertVisual('zen_kana')<CR>

vnoremap <silent> gHH <C-\><C-N>:call <SID>HzjaConvertVisual('han_ascii')<CR>
vnoremap <silent> gZZ <C-\><C-N>:call <SID>HzjaConvertVisual('zen_kana')<CR>

if has('gui_running')
  vnoremenu 1.120 PopUp.-SEP3-	<Nop>
  vnoremenu 1.130.100 PopUp.Spp(&H).S(&L) <C-\><C-N>:call <SID>HzjaConvertVisual('han_all')<CR>
  vnoremenu 1.130.110 PopUp.Spp(&H).ASCII(&A) <C-\><C-N>:call <SID>HzjaConvertVisual('han_ascii')<CR>
  vnoremenu 1.130.120 PopUp.Spp(&H).L(&M) <C-\><C-N>:call <SID>HzjaConvertVisual('han_kigou')<CR>
  vnoremenu 1.130.130 PopUp.Spp(&H).p(&W) <C-\><C-N>:call <SID>HzjaConvertVisual('han_eisu')<CR>
  vnoremenu 1.130.140 PopUp.Spp(&H).J^Ji(&J) <C-\><C-N>:call <SID>HzjaConvertVisual('han_kana')<CR>
  vnoremenu 1.140.100 PopUp.pSp(&Z).S(&L) <C-\><C-N>:call <SID>HzjaConvertVisual('zen_all')<CR>
  vnoremenu 1.140.110 PopUp.pSp(&Z).ASCII(&A) <C-\><C-N>:call <SID>HzjaConvertVisual('zen_ascii')<CR>
  vnoremenu 1.140.120 PopUp.pSp(&Z).L(&M) <C-\><C-N>:call <SID>HzjaConvertVisual('zen_kigou')<CR>
  vnoremenu 1.140.130 PopUp.pSp(&Z).p(&W) <C-\><C-N>:call <SID>HzjaConvertVisual('zen_eisu')<CR>
  vnoremenu 1.140.140 PopUp.pSp(&Z).J^Ji(&J) <C-\><C-N>:call <SID>HzjaConvertVisual('zen_kana')<CR>
endif

function! HzjaConvertComplete(argleand, cmdline, curpos)
  call s:Initialize()
  return s:targetlist
endfunction

function! s:HzjaConvert(target) range
  let nline = a:firstline
  while nline <= a:lastline
    call setline(nline, HzjaConvert(getline(nline), a:target))
    let nline = nline + 1
  endwhile
endfunction

" ^ꂽ̔pSp𑊌݂ɕϊBϊ̕@͈targetŕ
" ƂĎw肷Bwł镶͈ȉ̂ƂB
"
"   han_all	SĂ̑Spp
"   han_ascii	SpAXL[p
"   han_kana	SpJ^Jip
"   han_eisu	Sppp
"   han_kigou	SpLp
"   zen_all	SĂ̔pSp
"   zen_ascii	pAXL[Sp
"   zen_kana	pJ^JiSp
"   zen_eisu	ppSp
"   zen_kigou	pLSp
"
function! HzjaConvert(line, target)
  call s:Initialize()
  if !exists('s:mx_'.a:target)
    return a:line
  else
    let mode = a:target =~# '^han_' ? 'Hankaku' : 'Zenkaku'
    return substitute(a:line, s:mx_{a:target}, '\=s:ToggleLine(submatch(0),0,0,mode)', 'g')
  endif
endfunction

function! s:HzjaConvertVisual(target)
  call s:Initialize()
  let save_regcont = @"
  let save_regtype = getregtype('"')
  normal! gvy
  call setreg('"', HzjaConvert(@", a:target), getregtype('"'))
  normal! gv""p
  call setreg('"', save_regcont, save_regtype)
endfunction

let s:init = 0
function! s:Initialize()
  if s:init != 0
    return
  endif
  let s:init = 1

  let s:match_character = '\%([]\|[]\|.\)'
  let s:match_hankaku = '\%([]\|[]\|[ -~]\)'

  let zen_ascii = '@Ihfij{C|D^OPQRSTUVWXFGH`abcdefghijklmnopqrstuvwxymnOQeobp`'
  let zen_kana = 'BuvA@BDFHb[ACEGIJLNPRTVXZ\^`cegijklmnqtwz}~JKKMOQSUWY[]_adfhorux{psvy|'
  let han_ascii = " !\"#$%&'()*+,\\-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~"
  let han_kana = ''
  let s:mx_han_all = "[".zen_ascii.zen_kana."]\\+"
  let s:mx_zen_all = "[".han_ascii.han_kana."]\\+"
  let s:mx_han_ascii = "[".zen_ascii."]\\+"
  let s:mx_zen_ascii = "[".han_ascii."]\\+"
  let s:mx_han_kana = "[".zen_kana."]\\+"
  let s:mx_zen_kana = "[".han_kana."]\\+"
  let s:mx_han_eisu = '[OPQRSTUVWX`abcdefghijklmnopqrstuvwxy]\+'
  let s:mx_zen_eisu = '[0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz]\+'
  let s:mx_han_kigou = '[Ihfij{C|D^FGHmnOQeobp`]\+'
  let s:mx_zen_kigou = "[!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_`{|}~]\\+"
  let s:targetlist = "han_all\<NL>zen_all\<NL>han_ascii\<NL>zen_ascii\<NL>han_kana\<NL>zen_kana\<NL>han_eisu\<NL>zen_eisu\<NL>han_kigou\<NL>zen_kigou"

  " pSpe[u쐬
  let tmp = ''
  let tmp = tmp . " @!I\"h#$%&'f(i)j*+{,C-|.D/^"
  let tmp = tmp . '0O1P2Q3R4S5T6U7V8W9X:F;G<=>?H'
  let tmp = tmp . '@A`BaCbDcEdFeGfHgIhJiKjLkMlNmOn'
  let tmp = tmp . 'PoQpRqSrTsUtVuWvXwYxZy[m\]n^O_Q'
  let tmp = tmp . '`eabcdefghijklmno'
  let tmp = tmp . 'pqrstuvwxyz{o|b}p~`'
  let tmp = tmp . 'BuvA@BDFHb'
  let tmp = tmp . '[ACEGIJLNPRTVXZ\'
  let tmp = tmp . '^`cÃeăgŃiƃjǃkȃlɃmʃn˃q̃t̓w΃zσ}'
  let tmp = tmp . 'Ѓ~у҃ӃԃՃփ׃؃كڃۃ܃݃ށJ߁K'
  let tmp = tmp . 'ރރKރMރOރQރSރUރWރYރ[ރ]'
  let tmp = tmp . 'ރ_ރaރdރfރhރoރrރuރxރ{'
  let tmp = tmp . '߃p߃s߃v߃y߃|'
  let tmp = tmp . ''
  let s:table_h2z = tmp

  " Sppϊe[u쐬B
  let s:table_z2h = ''
  let startcol = 0
  let endcol = strlen(s:table_h2z)
  let curcol = 0
  let mx = '^\(' . s:match_hankaku . '\)\(.\)'
  while curcol < endcol
    let char = matchstr(s:table_h2z, mx, curcol)
    let s:table_z2h = s:table_z2h . substitute(char, mx, '\2\1', '')
    let curcol = curcol + strlen(char)
  endwhile
endfunction

"
" R}hŎw肳ꂽ̈ϊ
"
function! s:ToggleLineWise(operator) range
  call s:Initialize()

  let ncurline = a:firstline
  while ncurline <= a:lastline
    call setline(ncurline, s:ToggleLine(getline(ncurline), 0, 0, a:operator))
    let ncurline = ncurline + 1
  endwhile
endfunction

"
" ^ꂽϊĕԂB
"
function! s:ToggleLine(line, startcolumn, endcolumn, operator)
  let endcol = ((a:endcolumn > 0 && a:endcolumn < strlen(a:line))? a:endcolumn : strlen(a:line)) - 1
  let startcol = a:startcolumn > 0 ? a:startcolumn - 1: 0
  let curcol = startcol
  let newline = ''
  while curcol <= endcol
    let char = matchstr(a:line, s:match_character, curcol)
    let newline = newline . s:{a:operator}Char(char)
    let curcol = curcol + strlen(char)
  endwhile
  return strpart(a:line, 0, startcol) . newline . strpart(a:line, curcol)
endfunction

function! ToHankaku(str)
  call s:Initialize()
  return s:ToggleLine(a:str, 0, 0, 'Hankaku')
endfunction

function! ToZenkaku(str)
  call s:Initialize()
  return s:ToggleLine(a:str, 0, 0, 'Zenkaku')
endfunction

"
" char\ȂΔp/SpϊĕԂBϊłȂꍇ͂̂܂܁B
"
function! s:ToggleChar(char)
  return (s:IsHankaku(a:char)) ? (s:ZenkakuChar(a:char)) : (s:HankakuChar(a:char))
endfunction

"
" char\ȂΑSpɕϊĕԂBϊłȂꍇ͂̂܂܁B
"
function! s:ZenkakuChar(char)
  if s:IsHankaku(a:char)
    let pos = matchend(s:table_h2z, '\V\C' . escape(a:char, '\'))
    if pos >= 0
      return matchstr(s:table_h2z, '.', pos)
    endif
  endif
  return a:char
endfunction

"
" char\ȂΔpɕϊĕԂBϊłȂꍇ͂̂܂܁B
"
function! s:HankakuChar(char)
  if !s:IsHankaku(a:char)
    let pos = matchend(s:table_z2h, '\V\C' . escape(a:char, '\'))
    if pos >= 0
      return matchstr(s:table_z2h, s:match_hankaku, pos)
    endif
  endif
  return a:char
endfunction

"
" ^ꂽpǂ𔻒肷B
"
function! s:IsHankaku(char)
  return a:char =~ '^' . s:match_hankaku . '$'
endfunction
